// Copyright 2017 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package com.google.devtools.build.lib.buildeventservice;

import static com.google.common.truth.Truth.assertThat;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.devtools.build.lib.testutil.ManualClock;
import com.google.devtools.build.v1.BuildEvent;
import com.google.devtools.build.v1.BuildEvent.BuildComponentStreamFinished;
import com.google.devtools.build.v1.BuildEvent.BuildComponentStreamFinished.FinishType;
import com.google.devtools.build.v1.BuildEvent.BuildEnqueued;
import com.google.devtools.build.v1.BuildEvent.BuildFinished;
import com.google.devtools.build.v1.BuildEvent.InvocationAttemptFinished;
import com.google.devtools.build.v1.BuildEvent.InvocationAttemptStarted;
import com.google.devtools.build.v1.BuildStatus;
import com.google.devtools.build.v1.BuildStatus.Result;
import com.google.devtools.build.v1.OrderedBuildEvent;
import com.google.devtools.build.v1.PublishBuildToolEventStreamRequest;
import com.google.devtools.build.v1.PublishLifecycleEventRequest;
import com.google.devtools.build.v1.PublishLifecycleEventRequest.ServiceLevel;
import com.google.devtools.build.v1.StreamId;
import com.google.devtools.build.v1.StreamId.BuildComponent;
import com.google.protobuf.Any;
import com.google.protobuf.util.Timestamps;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

/** Tests {@link BuildEventServiceProtoUtil}. * */
@RunWith(JUnit4.class)
public class BuildEventServiceProtoUtilTest {

  private static final String BUILD_REQUEST_ID = "feedbeef-dead-4321-beef-deaddeaddead";
  private static final String BUILD_INVOCATION_ID = "feedbeef-dead-4444-beef-deaddeaddead";
  private static final String PROJECT_ID = "my_project";
  private static final String COMMAND_NAME = "test";
  private static final String ADDITIONAL_KEYWORD = "keyword=foo";
  private static final ImmutableList<String> EXPECTED_KEYWORDS =
      ImmutableList.of("command_name=" + COMMAND_NAME, "protocol_name=BEP", ADDITIONAL_KEYWORD);
  private final ManualClock clock = new ManualClock();
  private final BuildEventServiceProtoUtil besProtocol =
      new BuildEventServiceProtoUtil(
          BUILD_REQUEST_ID,
          BUILD_INVOCATION_ID,
          PROJECT_ID,
          COMMAND_NAME,
          clock,
          ImmutableSet.of(ADDITIONAL_KEYWORD));

  @Test
  public void testBuildEnqueued() {
    long expected = clock.advanceMillis(100);
    assertThat(besProtocol.buildEnqueued())
        .isEqualTo(
            PublishLifecycleEventRequest.newBuilder()
                .setServiceLevel(ServiceLevel.INTERACTIVE)
                .setProjectId(PROJECT_ID)
                .setBuildEvent(
                    OrderedBuildEvent.newBuilder()
                        .setStreamId(
                            StreamId.newBuilder()
                                .setBuildId(BUILD_REQUEST_ID)
                                .setComponent(BuildComponent.CONTROLLER))
                        .setSequenceNumber(1)
                        .setEvent(
                            BuildEvent.newBuilder()
                                .setEventTime(Timestamps.fromMillis(expected))
                                .setBuildEnqueued(BuildEnqueued.newBuilder())))
                .build());
  }

  @Test
  public void testInvocationAttemptStarted() {
    long expected = clock.advanceMillis(100);
    assertThat(besProtocol.invocationStarted())
        .isEqualTo(
            PublishLifecycleEventRequest.newBuilder()
                .setServiceLevel(ServiceLevel.INTERACTIVE)
                .setProjectId(PROJECT_ID)
                .setBuildEvent(
                    OrderedBuildEvent.newBuilder()
                        .setStreamId(
                            StreamId.newBuilder()
                                .setBuildId(BUILD_REQUEST_ID)
                                .setInvocationId(BUILD_INVOCATION_ID)
                                .setComponent(BuildComponent.CONTROLLER))
                        .setSequenceNumber(1)
                        .setEvent(
                            BuildEvent.newBuilder()
                                .setEventTime(Timestamps.fromMillis(expected))
                                .setInvocationAttemptStarted(
                                    InvocationAttemptStarted.newBuilder().setAttemptNumber(1))))
                .build());
  }

  @Test
  public void testInvocationAttemptFinished() {
    long expected = clock.advanceMillis(100);
    assertThat(besProtocol.invocationFinished(Result.COMMAND_SUCCEEDED))
        .isEqualTo(
            PublishLifecycleEventRequest.newBuilder()
                .setServiceLevel(ServiceLevel.INTERACTIVE)
                .setProjectId(PROJECT_ID)
                .setBuildEvent(
                    OrderedBuildEvent.newBuilder()
                        .setStreamId(
                            StreamId.newBuilder()
                                .setBuildId(BUILD_REQUEST_ID)
                                .setInvocationId(BUILD_INVOCATION_ID)
                                .setComponent(BuildComponent.CONTROLLER))
                        .setSequenceNumber(2)
                        .setEvent(
                            BuildEvent.newBuilder()
                                .setEventTime(Timestamps.fromMillis(expected))
                                .setInvocationAttemptFinished(
                                    InvocationAttemptFinished.newBuilder()
                                        .setInvocationStatus(
                                            BuildStatus.newBuilder()
                                                .setResult(Result.COMMAND_SUCCEEDED)))))
                .build());
  }

  @Test
  public void testBuildFinished() {
    long expected = clock.advanceMillis(100);
    assertThat(besProtocol.buildFinished(Result.COMMAND_SUCCEEDED))
        .isEqualTo(
            PublishLifecycleEventRequest.newBuilder()
                .setServiceLevel(ServiceLevel.INTERACTIVE)
                .setProjectId(PROJECT_ID)
                .setBuildEvent(
                    OrderedBuildEvent.newBuilder()
                        .setStreamId(
                            StreamId.newBuilder()
                                .setBuildId(BUILD_REQUEST_ID)
                                .setComponent(BuildComponent.CONTROLLER))
                        .setSequenceNumber(2)
                        .setEvent(
                            BuildEvent.newBuilder()
                                .setEventTime(Timestamps.fromMillis(expected))
                                .setBuildFinished(
                                    BuildFinished.newBuilder()
                                        .setStatus(
                                            BuildStatus.newBuilder()
                                                .setResult(Result.COMMAND_SUCCEEDED)))))
                .build());
  }

  @Test
  public void testStreamEvents() {
    long firstEventTimestamp = clock.advanceMillis(100);
    Any anything = Any.getDefaultInstance();
    assertThat(besProtocol.bazelEvent(1, anything))
        .isEqualTo(
            PublishBuildToolEventStreamRequest.newBuilder()
                .addAllNotificationKeywords(EXPECTED_KEYWORDS)
                .setOrderedBuildEvent(
                    OrderedBuildEvent.newBuilder()
                        .setStreamId(
                            StreamId.newBuilder()
                                .setBuildId(BUILD_REQUEST_ID)
                                .setInvocationId(BUILD_INVOCATION_ID)
                                .setComponent(BuildComponent.TOOL))
                        .setSequenceNumber(1)
                        .setEvent(
                            BuildEvent.newBuilder()
                                .setEventTime(Timestamps.fromMillis(firstEventTimestamp))
                                .setBazelEvent(anything))
                        .build())
                .build());

    long secondEventTimestamp = clock.advanceMillis(100);
    assertThat(besProtocol.bazelEvent(2, anything))
        .isEqualTo(
            PublishBuildToolEventStreamRequest.newBuilder()
                .setOrderedBuildEvent(
                    OrderedBuildEvent.newBuilder()
                        .setStreamId(
                            StreamId.newBuilder()
                                .setBuildId(BUILD_REQUEST_ID)
                                .setInvocationId(BUILD_INVOCATION_ID)
                                .setComponent(BuildComponent.TOOL))
                        .setSequenceNumber(2)
                        .setEvent(
                            BuildEvent.newBuilder()
                                .setEventTime(Timestamps.fromMillis(secondEventTimestamp))
                                .setBazelEvent(anything))
                        .build())
                .build());

    long thirdEventTimestamp = clock.advanceMillis(100);
    assertThat(besProtocol.streamFinished(3))
        .isEqualTo(
            PublishBuildToolEventStreamRequest.newBuilder()
                .setOrderedBuildEvent(
                    OrderedBuildEvent.newBuilder()
                        .setStreamId(
                            StreamId.newBuilder()
                                .setBuildId(BUILD_REQUEST_ID)
                                .setInvocationId(BUILD_INVOCATION_ID)
                                .setComponent(BuildComponent.TOOL))
                        .setSequenceNumber(3)
                        .setEvent(
                            BuildEvent.newBuilder()
                                .setEventTime(Timestamps.fromMillis(thirdEventTimestamp))
                                .setComponentStreamFinished(
                                    BuildComponentStreamFinished.newBuilder()
                                        .setType(FinishType.FINISHED)))
                        .build())
                .build());
  }
}
