pub fn instruction_str(opcode: u8) -> String {
    match opcode {
        0x00 => "NOP".into(),
        0x01 => "LD BC, u16".into(),
        0x02 => "LD (BC), A".into(),
        0x03 => "INC BC".into(),
        0x04 => "INC B".into(),
        0x05 => "DEC B".into(),
        0x06 => "LD B, u8".into(),
        0x07 => "RLCA".into(),
        0x08 => "LD (u16), SP".into(),
        0x09 => "ADD HL, BC".into(),
        0x0A => "LD A, (BC)".into(),
        0x0B => "DEC BC".into(),
        0x0C => "INC C".into(),
        0x0D => "DEC C".into(),
        0x0E => "LD C, u8".into(),
        0x0F => "RRCA".into(),
        0x10 => "STOP".into(),
        0x11 => "LD DE, u16".into(),
        0x12 => "LD (DE), A".into(),
        0x13 => "INC DE".into(),
        0x14 => "INC D".into(),
        0x15 => "DEC D".into(),
        0x16 => "LD D, u8".into(),
        0x17 => "RLA".into(),
        0x18 => "JR i8".into(),
        0x19 => "ADD HL, DE".into(),
        0x1A => "LD A, (DE)".into(),
        0x1B => "DEC DE".into(),
        0x1C => "INC E".into(),
        0x1D => "DEC E".into(),
        0x1E => "LD E, u8".into(),
        0x1F => "RRA".into(),
        0x20 => "JR NZ, i8".into(),
        0x21 => "LD HL, u16".into(),
        0x22 => "LD (HL+), A".into(),
        0x23 => "INC HL".into(),
        0x24 => "INC H".into(),
        0x25 => "DEC H".into(),
        0x26 => "LD H, u8".into(),
        0x27 => "DAA".into(),
        0x28 => "JR Z, i8".into(),
        0x29 => "ADD HL, HL".into(),
        0x2A => "LD A, (HL+)".into(),
        0x2B => "DEC HL".into(),
        0x2C => "INC L".into(),
        0x2D => "DEC L".into(),
        0x2E => "LD L, u8".into(),
        0x2F => "CPL".into(),
        0x30 => "JR NC, i8".into(),
        0x31 => "LD SP, u16".into(),
        0x32 => "LD (HL-), A".into(),
        0x33 => "INC SP".into(),
        0x34 => "INC (HL)".into(),
        0x35 => "DEC (HL)".into(),
        0x36 => "LD (HL), u8".into(),
        0x37 => "SCF".into(),
        0x38 => "JR C, i8".into(),
        0x39 => "ADD HL, SP".into(),
        0x3A => "LD A, (HL-)".into(),
        0x3B => "DEC SP".into(),
        0x3C => "INC A".into(),
        0x3D => "DEC A".into(),
        0x3E => "LD A, u8".into(),
        0x3F => "CCF".into(),
        0x40..=0x7F => {
            if opcode == 0x76 {
                "HALT".into()
            } else {
                let r = register_bits_to_str(opcode >> 3);
                let rp = register_bits_to_str(opcode);
                format!("LD {r}, {rp}")
            }
        }
        0x80..=0x87 => format!("ADD A, {}", register_bits_to_str(opcode)),
        0x88..=0x8F => format!("ADC A, {}", register_bits_to_str(opcode)),
        0x90..=0x97 => format!("SUB A, {}", register_bits_to_str(opcode)),
        0x98..=0x9F => format!("SBC A, {}", register_bits_to_str(opcode)),
        0xA0..=0xA7 => format!("AND A, {}", register_bits_to_str(opcode)),
        0xA8..=0xAF => format!("XOR A, {}", register_bits_to_str(opcode)),
        0xB0..=0xB7 => format!("OR A, {}", register_bits_to_str(opcode)),
        0xB8..=0xBF => format!("CP A, {}", register_bits_to_str(opcode)),
        0xC0 => "RET NZ".into(),
        0xC1 => "POP BC".into(),
        0xC2 => "JP NZ, u16".into(),
        0xC3 => "JP u16".into(),
        0xC4 => "CALL NZ, u16".into(),
        0xC5 => "PUSH BC".into(),
        0xC6 => "ADD A, u8".into(),
        0xC7 => "RST $00".into(),
        0xC8 => "RET Z".into(),
        0xC9 => "RET".into(),
        0xCA => "JP Z, u16".into(),
        0xCB => "(CB prefix)".into(),
        0xCC => "CALL Z, u16".into(),
        0xCD => "CALL u16".into(),
        0xCE => "ADC A, u8".into(),
        0xCF => "RST $08".into(),
        0xD0 => "RET NC".into(),
        0xD1 => "POP DE".into(),
        0xD2 => "JP NC, u16".into(),
        0xD4 => "CALL NC, u16".into(),
        0xD5 => "PUSH DE".into(),
        0xD6 => "SUB A, u8".into(),
        0xD7 => "RST $10".into(),
        0xD8 => "RET C".into(),
        0xD9 => "RETI".into(),
        0xDA => "JP C, u16".into(),
        0xDC => "CALL C, u16".into(),
        0xDE => "SBC A, u8".into(),
        0xDF => "RST $18".into(),
        0xE0 => "LDH (u8), A".into(),
        0xE1 => "POP HL".into(),
        0xE2 => "LD ($FF00+C), A".into(),
        0xE5 => "PUSH HL".into(),
        0xE6 => "AND A, u8".into(),
        0xE7 => "RST $20".into(),
        0xE8 => "ADD SP, i8".into(),
        0xE9 => "JP HL".into(),
        0xEA => "LD (u16), A".into(),
        0xEE => "XOR A, u8".into(),
        0xEF => "RST $28".into(),
        0xF0 => "LDH A, (u8)".into(),
        0xF1 => "POP AF".into(),
        0xF2 => "LD A, ($FF00+C)".into(),
        0xF3 => "DI".into(),
        0xF5 => "PUSH AF".into(),
        0xF6 => "OR A, u8".into(),
        0xF7 => "RST $30".into(),
        0xF8 => "LD HL, SP+i8".into(),
        0xF9 => "LD SP, HL".into(),
        0xFA => "LD A, (u16)".into(),
        0xFB => "EI".into(),
        0xFE => "CP A, u8".into(),
        0xFF => "RST $38".into(),
        _ => "(invalid opcode)".into(),
    }
}

fn register_bits_to_str(register_bits: u8) -> &'static str {
    match register_bits & 0x7 {
        0x0 => "B",
        0x1 => "C",
        0x2 => "D",
        0x3 => "E",
        0x4 => "H",
        0x5 => "L",
        0x6 => "(HL)",
        0x7 => "A",
        _ => unreachable!("value & 0x7 is always <= 0x7"),
    }
}

pub fn cb_instruction_str(opcode: u8) -> String {
    match opcode {
        0x00..=0x07 => format!("RLC {}", register_bits_to_str(opcode)),
        0x08..=0x0F => format!("RRC {}", register_bits_to_str(opcode)),
        0x10..=0x17 => format!("RL {}", register_bits_to_str(opcode)),
        0x18..=0x1F => format!("RR {}", register_bits_to_str(opcode)),
        0x20..=0x27 => format!("SLA {}", register_bits_to_str(opcode)),
        0x28..=0x2F => format!("SRA {}", register_bits_to_str(opcode)),
        0x30..=0x37 => format!("SWAP {}", register_bits_to_str(opcode)),
        0x38..=0x3F => format!("SRL {}", register_bits_to_str(opcode)),
        0x40..=0x7F => format!("BIT {}, {}", (opcode >> 3) & 7, register_bits_to_str(opcode)),
        0x80..=0xBF => format!("RES {}, {}", (opcode >> 3) & 7, register_bits_to_str(opcode)),
        0xC0..=0xFF => format!("SET {}, {}", (opcode >> 3) & 7, register_bits_to_str(opcode)),
    }
}
