package cmd

import (
	"bufio"
	"bytes"
	"errors"
	"strings"
	"testing"
)

func TestRebaser_RebaseInteractive_SelectValid(t *testing.T) {
	var buf bytes.Buffer

	mockClient := &mockAddGitClient{}
	r := &Rebaser{
		gitClient:    mockClient,
		outputWriter: &buf,
		helper:       NewHelper(),

		inputReader: bufio.NewReader(strings.NewReader("2\n")),
	}
	r.helper.outputWriter = &buf

	r.RebaseInteractive()

	output := buf.String()
	if !strings.Contains(output, "Current branch: feature/test") {
		t.Errorf("expected branch name, got: %s", output)
	}
	if !strings.Contains(output, "Select number of commits to rebase") {
		t.Errorf("expected prompt, got: %s", output)
	}
	if !strings.Contains(output, "Rebase successful") {
		t.Errorf("expected success message, got: %s", output)
	}
}

func TestRebaser_RebaseInteractive_BranchError(t *testing.T) {
	var buf bytes.Buffer
	mockClient := &mockAddGitClient{}
	// Override GetCurrentBranch to return error
	mockClient.GetCurrentBranchFunc = func() (string, error) {
		return "", errors.New("failed to get current branch")
	}
	r := &Rebaser{
		gitClient:    mockClient,
		outputWriter: &buf,
		helper:       NewHelper(),

		inputReader: bufio.NewReader(strings.NewReader("1\n")),
	}
	r.helper.outputWriter = &buf

	r.RebaseInteractive()

	output := buf.String()
	if !strings.Contains(output, "Error: failed to get current branch") {
		t.Errorf("expected branch error message, got: %s", output)
	}
}

func TestRebaser_RebaseInteractive_Cancel(t *testing.T) {
	var buf bytes.Buffer
	mockClient := &mockAddGitClient{}
	r := &Rebaser{
		gitClient:    mockClient,
		outputWriter: &buf,
		helper:       NewHelper(),
		inputReader:  bufio.NewReader(strings.NewReader("\n")),
	}
	r.helper.outputWriter = &buf

	r.RebaseInteractive()

	output := buf.String()
	if !strings.Contains(output, "Error: operation cancelled") {
		t.Errorf("expected cancellation message, got: %s", output)
	}
}

func TestRebaser_RebaseInteractive_InvalidNumber(t *testing.T) {
	var buf bytes.Buffer
	mockClient := &mockAddGitClient{}
	r := &Rebaser{
		gitClient:    mockClient,
		outputWriter: &buf,
		helper:       NewHelper(),
		inputReader:  bufio.NewReader(strings.NewReader("abc\n")),
	}
	r.helper.outputWriter = &buf

	r.RebaseInteractive()

	output := buf.String()
	if !strings.Contains(output, "Error: invalid number") {
		t.Errorf("expected invalid number message, got: %s", output)
	}
}

func TestRebaser_RebaseInteractive_NoHistory(t *testing.T) {
	var buf bytes.Buffer
	mockClient := &mockAddGitClient{}
	// Override LogOneline to return empty history
	mockClient.LogOnelineFunc = func(_, _ string) (string, error) {
		return "", nil
	}
	r := &Rebaser{
		gitClient:    mockClient,
		outputWriter: &buf,
		helper:       NewHelper(),
		inputReader:  bufio.NewReader(strings.NewReader("1\n")),
	}
	r.helper.outputWriter = &buf

	r.RebaseInteractive()

	output := buf.String()
	if !strings.Contains(output, "Error: no commit history found") {
		t.Errorf("expected no history message, got: %s", output)
	}
}

func TestRebaser_RebaseInteractive_LogError(t *testing.T) {
	var buf bytes.Buffer

	mockClient := &mockAddGitClient{}
	// Override LogOneline to return error
	mockClient.LogOnelineFunc = func(_, _ string) (string, error) {
		return "", errors.New("failed to get git log")
	}
	r := &Rebaser{
		gitClient:    mockClient,
		outputWriter: &buf,
		helper:       NewHelper(),
		inputReader:  bufio.NewReader(strings.NewReader("1\n")),
	}
	r.helper.outputWriter = &buf

	r.RebaseInteractive()

	output := buf.String()
	if !strings.Contains(output, "Error: failed to get git log") {
		t.Errorf("expected log error message, got: %s", output)
	}
}

func TestRebaser_RebaseInteractive_RebaseError(t *testing.T) {
	var buf bytes.Buffer

	mockClient := &mockAddGitClient{}
	// Override RebaseInteractive to return error
	mockClient.RebaseInteractiveFunc = func(_ int) error {
		return errors.New("rebase failed")
	}
	r := &Rebaser{
		gitClient:    mockClient,
		outputWriter: &buf,
		helper:       NewHelper(),
		inputReader:  bufio.NewReader(strings.NewReader("1\n")),
	}
	r.helper.outputWriter = &buf

	r.RebaseInteractive()

	output := buf.String()
	if !strings.Contains(output, "Error: rebase failed") {
		t.Errorf("expected rebase error message, got: %s", output)
	}
}

func TestRebaser_Rebase(t *testing.T) {
	cases := []struct {
		name           string
		args           []string
		expectedOutput string
		mockInput      string
	}{
		{
			name:           "interactive rebase via -i",
			args:           []string{"-i"},
			expectedOutput: "Current branch:",
			mockInput:      "1\n",
		},
		{
			name:           "no args",
			args:           []string{},
			expectedOutput: "Usage: ggc rebase",
		},
		{
			name:           "invalid command",
			args:           []string{"invalid"},
			expectedOutput: "Usage: ggc rebase",
		},
	}

	for _, tc := range cases {
		t.Run(tc.name, func(t *testing.T) {
			var buf bytes.Buffer
			mockClient := &mockAddGitClient{}
			r := &Rebaser{
				gitClient:    mockClient,
				outputWriter: &buf,
				helper:       NewHelper(),
				inputReader:  bufio.NewReader(strings.NewReader(tc.mockInput)),
			}
			r.helper.outputWriter = &buf

			r.Rebase(tc.args)

			output := buf.String()
			if !strings.Contains(output, tc.expectedOutput) {
				t.Errorf("expected output to contain %q, got %q", tc.expectedOutput, output)
			}
		})
	}
}

func TestRebaser_Rebase_InteractiveCancel(t *testing.T) {
	var buf bytes.Buffer
	mockClient := &mockAddGitClient{}
	// Override LogOneline to return empty history
	mockClient.LogOnelineFunc = func(_, _ string) (string, error) {
		return "", nil
	}
	r := &Rebaser{
		gitClient:    mockClient,
		outputWriter: &buf,
		helper:       NewHelper(),
		inputReader:  bufio.NewReader(strings.NewReader("\n")),
	}
	r.helper.outputWriter = &buf

	r.Rebase([]string{"-i"})

	output := buf.String()
	if !strings.Contains(output, "Error: no commit history found") {
		t.Errorf("expected no history message, got %q", output)
	}
}

func TestRebaser_Rebase_Error(t *testing.T) {
	var buf bytes.Buffer
	mockClient := &mockAddGitClient{}
	r := &Rebaser{
		gitClient:    mockClient,
		outputWriter: &buf,
		helper:       NewHelper(),
		inputReader:  bufio.NewReader(strings.NewReader("y\n")),
	}
	r.helper.outputWriter = &buf

	r.Rebase([]string{"-i"})

	output := buf.String()
	if !strings.Contains(output, "Error:") {
		t.Errorf("expected error message, got %q", output)
	}
}
