package checker_test

import (
	"testing"

	"github.com/stretchr/testify/require"
	"github.com/tufin/oasdiff/checker"
	"github.com/tufin/oasdiff/diff"
	"github.com/tufin/oasdiff/load"
)

// CL: removing an existing operation id
func TestOperationIdRemoved(t *testing.T) {
	s1, err := open("../data/checker/operation_id_removed_base.yaml")
	require.NoError(t, err)
	s2, err := open("../data/checker/operation_id_removed_base.yaml")
	require.NoError(t, err)

	s2.Spec.Paths.Value("/api/v1.0/groups").Post.OperationID = ""

	d, osm, err := diff.GetWithOperationsSourcesMap(diff.NewConfig(), s1, s2)
	require.NoError(t, err)
	errs := checker.CheckBackwardCompatibilityUntilLevel(singleCheckConfig(checker.APIOperationIdUpdatedCheck), d, osm, checker.INFO)
	require.Len(t, errs, 1)
	require.Equal(t, checker.ApiChange{
		Id:          checker.APIOperationIdRemovedId,
		Args:        []any{"createOneGroup", ""},
		Level:       checker.INFO,
		Operation:   "POST",
		Path:        "/api/v1.0/groups",
		Source:      load.NewSource("../data/checker/operation_id_removed_base.yaml"),
		OperationId: "createOneGroup",
	}, errs[0])
}

// CL: updating an existing operation id
func TestOperationIdUpdated(t *testing.T) {
	s1, err := open("../data/checker/operation_id_removed_base.yaml")
	require.NoError(t, err)
	s2, err := open("../data/checker/operation_id_removed_base.yaml")
	require.NoError(t, err)

	s2.Spec.Paths.Value("/api/v1.0/groups").Post.OperationID = "newOperationId"

	d, osm, err := diff.GetWithOperationsSourcesMap(diff.NewConfig(), s1, s2)
	require.NoError(t, err)
	errs := checker.CheckBackwardCompatibilityUntilLevel(singleCheckConfig(checker.APIOperationIdUpdatedCheck), d, osm, checker.INFO)
	require.Len(t, errs, 1)
	require.Equal(t, checker.ApiChange{
		Id:          checker.APIOperationIdRemovedId,
		Args:        []any{"createOneGroup", "newOperationId"},
		Level:       checker.INFO,
		Operation:   "POST",
		Path:        "/api/v1.0/groups",
		Source:      load.NewSource("../data/checker/operation_id_removed_base.yaml"),
		OperationId: "createOneGroup",
	}, errs[0])

	require.Equal(t, "api operation id 'createOneGroup' removed and replaced with 'newOperationId'", errs[0].GetUncolorizedText(checker.NewDefaultLocalizer()))
}

// CL: adding a new operation id
func TestOperationIdAdded(t *testing.T) {
	s1, err := open("../data/checker/operation_id_added_base.yaml")
	require.NoError(t, err)
	s2, err := open("../data/checker/operation_id_added_base.yaml")
	require.NoError(t, err)

	s2.Spec.Paths.Value("/api/v1.0/groups").Post.OperationID = "NewOperationId"

	d, osm, err := diff.GetWithOperationsSourcesMap(diff.NewConfig(), s1, s2)
	require.NoError(t, err)
	errs := checker.CheckBackwardCompatibilityUntilLevel(singleCheckConfig(checker.APIOperationIdUpdatedCheck), d, osm, checker.INFO)
	require.Len(t, errs, 1)
	require.Equal(t, checker.ApiChange{
		Id:          checker.APIOperationIdAddId,
		Args:        []any{"NewOperationId"},
		Level:       checker.INFO,
		Operation:   "POST",
		Path:        "/api/v1.0/groups",
		Source:      load.NewSource("../data/checker/operation_id_added_base.yaml"),
		OperationId: "NewOperationId",
	}, errs[0])

	require.Equal(t, "api operation id 'NewOperationId' was added", errs[0].GetUncolorizedText(checker.NewDefaultLocalizer()))
}
