package core

import (
	"testing"

	"github.com/nalgeon/be"
)

func TestValue(t *testing.T) {
	t.Run("bytes", func(t *testing.T) {
		v := Value([]byte("hello"))
		be.Equal(t, v.IsZero(), false)
		be.Equal(t, v.Bytes(), []byte("hello"))
		be.Equal(t, v.String(), "hello")
		_, err := v.Bool()
		be.Equal(t, err.Error(), `strconv.ParseBool: parsing "hello": invalid syntax`)
		_, err = v.Int()
		be.Equal(t, err.Error(), `strconv.Atoi: parsing "hello": invalid syntax`)
		_, err = v.Float()
		be.Equal(t, err.Error(), `strconv.ParseFloat: parsing "hello": invalid syntax`)
	})
	t.Run("string", func(t *testing.T) {
		v := Value("hello")
		be.Equal(t, v.IsZero(), false)
		be.Equal(t, v.Bytes(), []byte("hello"))
		be.Equal(t, v.String(), "hello")
		_, err := v.Bool()
		be.Equal(t, err.Error(), `strconv.ParseBool: parsing "hello": invalid syntax`)
		_, err = v.Int()
		be.Equal(t, err.Error(), `strconv.Atoi: parsing "hello": invalid syntax`)
		_, err = v.Float()
		be.Equal(t, err.Error(), `strconv.ParseFloat: parsing "hello": invalid syntax`)
	})
	t.Run("bool true", func(t *testing.T) {
		v := Value("1")
		be.Equal(t, v.IsZero(), false)
		be.Equal(t, v.Bytes(), []byte("1"))
		be.Equal(t, v.String(), "1")
		vbool, err := v.Bool()
		be.Err(t, err, nil)
		be.Equal(t, vbool, true)
		vint, err := v.Int()
		be.Err(t, err, nil)
		be.Equal(t, vint, 1)
		vfloat, err := v.Float()
		be.Err(t, err, nil)
		be.Equal(t, vfloat, 1.0)
	})
	t.Run("bool false", func(t *testing.T) {
		v := Value("0")
		be.Equal(t, v.IsZero(), false)
		be.Equal(t, v.Bytes(), []byte("0"))
		be.Equal(t, v.String(), "0")
		vbool, err := v.Bool()
		be.Err(t, err, nil)
		be.Equal(t, vbool, false)
		vint, err := v.Int()
		be.Err(t, err, nil)
		be.Equal(t, vint, 0)
		vfloat, err := v.Float()
		be.Err(t, err, nil)
		be.Equal(t, vfloat, 0.0)
	})
	t.Run("int", func(t *testing.T) {
		v := Value("42")
		be.Equal(t, v.IsZero(), false)
		be.Equal(t, v.Bytes(), []byte("42"))
		be.Equal(t, v.String(), "42")
		_, err := v.Bool()
		be.Equal(t, err.Error(), `strconv.ParseBool: parsing "42": invalid syntax`)
		vint, err := v.Int()
		be.Err(t, err, nil)
		be.Equal(t, vint, 42)
		vfloat, err := v.Float()
		be.Err(t, err, nil)
		be.Equal(t, vfloat, 42.0)
	})
	t.Run("float", func(t *testing.T) {
		v := Value("42.5")
		be.Equal(t, v.IsZero(), false)
		be.Equal(t, v.Bytes(), []byte("42.5"))
		be.Equal(t, v.String(), "42.5")
		_, err := v.Bool()
		be.Equal(t, err.Error(), `strconv.ParseBool: parsing "42.5": invalid syntax`)
		_, err = v.Int()
		be.Equal(t, err.Error(), `strconv.Atoi: parsing "42.5": invalid syntax`)
		vfloat, err := v.Float()
		be.Err(t, err, nil)
		be.Equal(t, vfloat, 42.5)
	})
	t.Run("empty string", func(t *testing.T) {
		v := Value("")
		be.Equal(t, v.IsZero(), true)
		be.Equal(t, v.Bytes(), []byte{})
		be.Equal(t, v.String(), "")
		vbool, err := v.Bool()
		be.Err(t, err, nil)
		be.Equal(t, vbool, false)
		vint, err := v.Int()
		be.Err(t, err, nil)
		be.Equal(t, vint, 0)
		vfloat, err := v.Float()
		be.Err(t, err, nil)
		be.Equal(t, vfloat, 0.0)
	})
	t.Run("nil", func(t *testing.T) {
		v := Value(nil)
		be.Equal(t, v.IsZero(), true)
		be.Equal(t, v.Bytes(), []byte(nil))
		be.Equal(t, v.String(), "")
		vbool, err := v.Bool()
		be.Err(t, err, nil)
		be.Equal(t, vbool, false)
		vint, err := v.Int()
		be.Err(t, err, nil)
		be.Equal(t, vint, 0)
		vfloat, err := v.Float()
		be.Err(t, err, nil)
		be.Equal(t, vfloat, 0.0)
	})
}
