# Release Process

This document explains how to create a new release of ngrep.

## Overview

ngrep uses **Git tags** to trigger release builds. When you push a version tag, GitHub Actions automatically:

1. Builds binaries for all supported platforms (9 platforms total)
2. Packages them as `.tar.gz` (Unix) or `.zip` (Windows)
3. Generates SHA256 checksums
4. Creates a GitHub Release with all artifacts
5. Builds and publishes Docker containers to GitHub Container Registry

## Supported Platforms

Release builds are created for:

- **Linux**: Ubuntu (x86_64, ARM64)
- **macOS**: macOS 15 (ARM64), macOS 26 (ARM64)
- **BSD**: FreeBSD 15, OpenBSD 7, NetBSD 10 (x86_64)
- **Solaris**: Solaris 11 (x86_64)
- **Windows**: Windows 11 (x86_64)

## Creating a Release

### 1. Prepare the Release

Ensure all changes are committed and pushed to `master`:

```bash
git checkout master
git pull
```

### 2. Update Version Information

**Version is now centralized in a single `VERSION` file.**

Update the version number:

```bash
# Edit the VERSION file (contains just the version number, e.g., "1.0.0")
echo "1.0.0" > VERSION

# Optionally update README.md header (line 1) to match
# Example: ## ngrep 1.0.0 (MM.DD.YYYY)

# Regenerate configure script from configure.ac
autoconf
```

**What gets updated automatically:**
- `config.h` (Unix builds) - Generated by `configure` from `VERSION`
- `ngrep` binary version string - Reads `VERSION` via `config.h`
- Windows builds - CMake reads `VERSION` file directly

**What you still update manually:**
- `README.md` line 1 - Update version and date in the header

Commit these changes:

```bash
git add VERSION README.md configure
git commit -m "Bump version to 1.0.0"
git push
```

### 3. Create and Push the Tag

Create an annotated tag with the version number:

```bash
# Format: vMAJOR.MINOR.PATCH
git tag -a v1.0.0 -m "Release 1.0.0"
git push origin v1.0.0
```

**Important**: The tag MUST start with `v` (e.g., `v1.0.0`, `v2.1.3`)

### 4. Monitor the Build

1. Go to: https://github.com/jpr5/ngrep/actions
2. Watch the "Release Build & Publish" workflow
3. Watch the "Docker Build & Publish" workflow
4. Build takes upwards of ~10 minutes (emulated BSD/Solaris VMs are slower)

### 5. Verify the Release

Once complete:

1. **Binary Release**:
   - Go to: https://github.com/jpr5/ngrep/releases
   - Verify all 9 platform artifacts are present
   - Check SHA256SUMS file
   - Test download and extraction of at least one artifact

2. **Docker Container**:
   - Go to: https://github.com/jpr5/ngrep/pkgs/container/ngrep
   - Verify the version tag is present (e.g., `1.0.0`)
   - Test pulling and running: `docker pull ghcr.io/jpr5/ngrep:1.0.0`

## Release Artifacts

Each release includes:

### Binary-Only Packages
- `ngrep-<platform>.tar.gz` - Just the `ngrep` binary
- `ngrep-windows-x86_64.zip` - Windows executable with DLLs

### Full Packages
- `ngrep-<platform>-full.tar.gz` - Complete installation (binary + man pages + docs)

### Checksums
- `SHA256SUMS` - SHA256 checksums for all artifacts

### Docker Containers
- `ghcr.io/jpr5/ngrep:1.0.0` - Specific version
- `ghcr.io/jpr5/ngrep:1.0` - Major.minor version
- `ghcr.io/jpr5/ngrep:1` - Major version
- `ghcr.io/jpr5/ngrep:latest` - Latest release
- Multi-architecture: linux/amd64, linux/arm64
- Alpine-based (~15-20MB compressed)

## Troubleshooting

### Build Failed for One Platform

If a single platform fails:
1. Check the workflow logs for that platform
2. Fix the issue in the code
3. Delete the tag: `git tag -d v1.0.0 && git push origin :refs/tags/v1.0.0`
4. Re-create the tag after fixing

### Wrong Version Number

If you tagged the wrong version:
1. Delete the GitHub Release (if created)
2. Delete the tag locally: `git tag -d v1.0.0`
3. Delete the tag remotely: `git push origin :refs/tags/v1.0.0`
4. Create the correct tag

### Pre-release / Beta Versions

For pre-releases, use tags like:
- `v1.0.0-beta1`
- `v1.0.0-rc1`
- `v2.0.0-alpha`

The workflow will still build them, but you should manually mark the GitHub Release as "pre-release".

## Version Management

ngrep uses a **centralized version system** with a single source of truth:

### Architecture

**Single Source**: `VERSION` file (contains just the version number, e.g., `1.0.0`)

**How it flows:**

1. **Unix/Linux/BSD/Solaris builds** (autoconf):
   - `configure.ac` reads `VERSION` via `m4_esyscmd_s([cat VERSION])`
   - `AC_INIT([ngrep],m4_esyscmd_s([cat VERSION]),...)`
   - `AC_DEFINE_UNQUOTED(VERSION, "AC_PACKAGE_VERSION", ...)` writes to `config.h`
   - `ngrep.c` includes `config.h` and uses `VERSION` macro

2. **Windows builds** (CMake):
   - `CMakeLists.txt` reads `VERSION` via `file(READ ... NGREP_VERSION)`
   - Passes `-DVERSION="1.0.0"` to compiler
   - `ngrep.c` uses `VERSION` macro from compile definition

3. **Documentation**:
   - `README.md` - Still updated manually (includes release date)

### Files Involved

- **`VERSION`** - Single source of truth (edit this to bump version)
- **`configure.ac`** - Reads `VERSION`, generates `config.h`
- **`ngrep.h`** - No longer contains version (removed)
- **`config.h`** - Generated file, contains `#define VERSION "..."`
- **`winXX/CMakeLists.txt`** - Reads `VERSION`, passes to compiler
- **`README.md`** - Manual update (version + date in header)

## Workflow Files

- `.github/workflows/matrix.yml` - Base build steps, reused by build & release
- `.github/workflows/build.yml` - CI validation (runs on every push)
- `.github/workflows/release.yml` - Release builds (runs on tags only)
- `.github/workflows/docker.yml` - Docker container builds (runs on push to master and tags)

## Notes

- **No artifacts on regular commits**: Only tagged releases create downloadable artifacts
- **Retention**: Workflow artifacts are kept for 5 days during build, then moved to GitHub Releases permanently
- **Permissions**: You need write access to the repository to push tags and create releases
- **Docker containers**: Published on both master commits (as `latest`) and version tags (as versioned tags)

## Distribution Channels

After a release, ngrep is available through:

1. **GitHub Releases**: https://github.com/jpr5/ngrep/releases
   - Platform-specific binaries
   - Full installation packages
   - SHA256 checksums

2. **GitHub Container Registry**: https://github.com/jpr5/ngrep/pkgs/container/ngrep
   - Docker containers for linux/amd64 and linux/arm64
   - Multiple version tags
   - Alpine-based for minimal size

3. **Source Code**: https://github.com/jpr5/ngrep
   - Git repository with full history
   - Build from source instructions
