# -*- coding: iso-8859-15 -*-
# This program is public domain
# Author: Paul Kienzle
u"""
Provides average mass for the elements:

*    mass, mass_units (u)
         The molar mass averaged over natural isotope abundance.

Provides mass and abundance information for isotopes:

*    mass, mass_units (u)
         The molar mass of the individual isotope.

*    abundance, abundance_units (%)
         Natural abundance for the isotope.

The atomic mass data for the isotopes and for the neutron were published
by Wang [#Wang]_ in the AME 2020 atomic mass evaluation (II).

Natural abundance mass and isotope fractions come from the IUPAC commission
on isotopic abundances and atomic weights (CIAAW) [#CIAAW]_. This is an
ill-defined problem since there is natural fractionation of the isotopes,
giving different masses and ratios from different samples. The current
recommended table therefore gives ranges of values for the natural mass
and isotope abundance rather than a single value with an uncertainty.

For periodictable, fractionation ranges for masses were replaced with abridged
standard atomic weights as given in Prohaska [#Prohaska]_. For the abundance
ratios the center value of the range was chosen. A few elements had to be
adjusted slightly so that they would sum to 100%.

The values for Ar and N were set to the values present in the atmosphere
and U was set to the values in a Namibian ore per the recommendations
in Meija [#Meija]_.

The values for Pb in the CIAAW table are too broad to be usable. For example,
206-Pb varies from 0.02 to 0.87 in monazite samples (Zhu 2020) [#Zhu]_. Rather
than return NaN for composition we replace the ranges with representative
atomic abundance values in Meija (2016). See the CIAAW page on
`lead <https://www.ciaaw.org/lead.htm>`_ for details.

Note: uncertainties are stored in `iso._mass_unc` and `iso._abundance_unc`,
but they are not yet part of the public interface.

.. [#Wang] Meng Wang et al. (2021) Chinese Phys. C 45 030003
    DOI:10.1088/1674-1137/abddaf
    From https://www-nds.iaea.org/amdc/ame2020/massround.mas20.txt (2023-07-06)
.. [#CIAAW] CIAAW. Isotopic compositions of the elements 2021.
   Available online at https://www.ciaaw.org.
.. [#Prohaska] T. Prohaska, et al. (2022)
    Standard atomic weights of the elements 2021.
    Pure Appl. Chem. 94. DOI:10.1515/pac-2019-0603
    From https://www.ciaaw.org/atomic-weights.htm (2023-07-06)
.. [#Meija] J. Meija et al. (2016)
    Isotopic compositions of the elements 2013
    Pure and Applied Chemistry 88, 293-306.
    From https://www.ciaaw.org/isotopic-abundances.htm (2023-07-06)
.. [#Zhu] Zhu, X., Benefield, J., Coplen, T., Gao, Z. & Holden, N. (2021).
    Variation of lead isotopic composition and atomic weight in terrestrial
    materials (IUPAC Technical Report). Pure and Applied Chemistry, 93(1), 155-166.
    https://doi.org/10.1515/pac-2018-0916
"""
from typing import cast
from .core import Element, Isotope, PeriodicTable, default_table
from .util import parse_uncertainty

def mass(atom: Element|Isotope) -> float:
    """
    Atomic weight.

    :Parameters:
        *isotope* : Isotope

    :Returns:
        *mass* : float | u
            Atomic weight of the element.
    """
    return atom._mass

def abundance(atom: Isotope) -> float:
    """
    Natural abundance.

    :Parameters:
        *isotope* : Isotope

    :Returns:
        *abundance* : float | %
    """
    return atom._abundance

def init(table: PeriodicTable, reload: bool=False) -> None:
    """Add mass attribute to period table elements and isotopes"""
    if 'mass' in table.properties and not reload:
        return
    table.properties.append('mass')
    Element.mass = property(mass, doc=mass.__doc__) # type: ignore[assignment]
    Element.mass_units = "u"
    Isotope.mass = property(mass, doc=mass.__doc__) # type: ignore[assignment]
    Isotope.abundance = property(abundance, doc=abundance.__doc__) # type: ignore[assignment]
    Isotope.abundance_units = "%"

    # Parse isotope mass table where each line looks like:
    #     z-el-iso,isotope mass(unc)#?,abundance(unc),element mass(unc)
    # The abundance and element masses will be set from other tables, so
    # ignore them here.
    for line in isotope_mass.split('\n'):
        isotope, iso_mass, iso_abundance, el_mass = line.split(',')
        zstr, sym, astr = isotope.split('-')
        el = table[int(zstr)]
        assert el.symbol == sym, \
            "Symbol %s does not match %s"%(sym, el.symbol)
        iso = el.add_isotope(int(astr))
        # Note: new mass table doesn't include nominal values for transuranics
        # so use old masses here and override later with new masses.
        el._mass, el._mass_unc = cast(tuple[float, float], parse_uncertainty(el_mass))
        #el._mass, el._mass_unc = None, None
        iso._mass, iso._mass_unc = cast(tuple[float, float], parse_uncertainty(iso_mass))
        #iso._abundance, iso._abundance_unc = parse_uncertainty(p)
        iso._abundance, iso._abundance_unc = 0, 0

    # # A single neutron is an isotope of element 0
    from .constants import neutron_mass, neutron_mass_unc
    el = table.n
    el._mass, el._mass_unc = neutron_mass, neutron_mass_unc
    iso = el.add_isotope(1)
    iso._mass, iso._mass_unc = neutron_mass, neutron_mass_unc
    iso._abundance, iso._abundance_unc = 0, 0

    # Parse element mass table where each line looks like:
    #    z  El  element mass(unc)|[low,high]|- note note ...
    for line in element_mass.split('\n'):
        zstr, symbol, name, valstr = line.split()[:4]
        #print(z, symbol, name, valstr)
        el = table[int(zstr)]
        if valstr != '-':
            #v, dv = parse_uncertainty(valstr)
            #delta = abs(v-el._mass)/el._mass*100
            #from uncertainties import ufloat as U
            #if delta > 0.01:
            #    print(f"{el.number}-{el.symbol} mass changed by {delta:.2f}% to {U(v,dv):fS} from {U(el._mass,el._mass_unc):fS}")
            el._mass, el._mass_unc = cast(tuple[float, float], parse_uncertainty(valstr))

    #Li_ratio = table.Li[7]._abundance/table.Li[6]._abundance

    # Parse isotope abundance table where lines look like:
    #    z  El element\n    iso mass(unc)|[low,high] note ...
    # Note: tables modified for Pb, Ar, and N to use 2013 values
    z = 0
    value: dict[int, tuple[float, float]] = {}
    for line in isotope_abundance.split('\n'):
        #print(line)
        # New element
        if line[0] not in ' \t':
            if z: # not the start
                #print(f"update z {z}-{table[z]}", value)
                el = table[z]
                # Find total so we can normalize fractions to sum to one.
                # This affects O, Mg, S, Hf by amounts less than the uncertainty.
                total = sum(v[0] for k, v in value.items())
                #if abs(total - 1) > 1e-10: print("total deviation for %s is %g"%(el, total-1))
                for z, pair in value.items():
                    iso = el[z]
                    #from uncertainties import ufloat as U
                    #v, dv = 100*pair[0]/total, 100*pair[1]/total
                    #delta = abs(v-iso._abundance)
                    #if delta >= 0.1  and v>1:
                    #    print(f"{el.number}-{el.symbol}-{z} abundance changed by {delta:.2f}% to {U(v,dv):fS} from {U(iso._abundance,iso._abundance_unc):fS}")

                    iso._abundance = 100*pair[0]/total
                    iso._abundance_unc = 100*pair[1]/total
            z = int(line.strip().split()[0])
            value = {}
            #print(f"new z {z}-{table[z]}")# from <{line}>")
        else:
            #print(line)
            parts = line.strip().split()
            #print(parts)
            value[int(parts[0])] = cast(tuple[float, float], parse_uncertainty(parts[1]))

    #new_Li_ratio = table.Li[7]._abundance/table.Li[6]._abundance
    #print(f"Li6:Li7 ratio changed from {Li_ratio:.1f} to {new_Li_ratio:.1f}")


def print_natural_mass(table: PeriodicTable|None=None) -> None:
    from uncertainties import ufloat as U  # type: ignore[import-untyped]
    table = default_table(table)
    for el in table:
        iso_mass = [
            U(iso.abundance, iso._abundance_unc)/100*U(iso.mass, iso._mass_unc)  # type: ignore[operator]
            for iso in el if iso.abundance>0]
        if iso_mass:
            el_mass = U(el.mass, el._mass_unc)
            iso_sum = sum(iso_mass)
            delta = el_mass - iso_sum # type: ignore[operator]
            # python 3.6 and above only
            if abs(delta.n) > 1e-3 or delta.s/iso_sum.n > 0.01:  # type: ignore[operator,union-attr]
                print(f"{el.number}-{el}: {el_mass:fS}, sum: {iso_sum:fS}, Δ={delta:fS}")
                #print(f"{el.number}-{el}: {delta:fS}")
            #print(f"{el.number}-{el}: {el_mass:fS}, sum: {iso_sum:fS}, Δ=")
            #print(f"{el.number}-{el}: {delta:fS}")
            #print("%d-%s: %s (from sum: %s)"%(el.number, el, str(el_mass), str(iso_sum)))

def print_abundance(table: PeriodicTable|None=None) -> None:
    table = default_table(table)
    for el in table:
        abundance = ["%8s %g"%(iso, iso.abundance/100) for iso in el if iso.abundance>0]
        if abundance:
            print("\n".join(abundance))
            print()

def check_abundance(table: PeriodicTable|None=None) -> None:
    table = default_table(table)
    for el in table:
        abundance = [iso.abundance for iso in el if iso.abundance>0]
        if abundance:
            assert abs(sum(abundance) - 100.0) < 1e-12,\
                "Inconsistent abundance for %d-%s: %g"%(el.number,el,sum(abundance))

# Table of masses.
# g Geological and biological materials are known in which the element has an
#   isotopic composition outside the limits for normal material. The difference
#   between the atomic weight of the element in such materials and that given in
#   the table may exceed the stated uncertainty.
# m Modified isotopic compositions may be found in commercially available
#   material because the material has been subjected to some undisclosed
#   or inadvertent isotopic fractionation. Substantial deviations in atomic
#   weight of the element from that given in the table can occur.
# r Range in isotopic composition of normal terrestrial material prevents
#   a more precise standard atomic weight being given; the tabulated value
#   and uncertainty should be applicable to normal material.
#
# Fractionation ranges have been replaced with the abridged standard weights
# as giving in Prohaska(2022).
#
# CIAAW. Isotopic compositions of the elements 2021. Available online at www.ciaaw.org.
#
# https://www.ciaaw.org/atomic-weights.htm (2023-07-06)

#Z 	Symbol 	Element 	Standard Atomic Weight 	Notes
element_mass = """\
1	H	hydrogen	  1.0080(2) [1.00784,1.00811]	m
2	He	helium	   4.002602(2)	g r
3	Li	lithium	  6.94(6)   [6.938,6.997]	m
4	Be	beryllium	   9.0121831(5)
5	B	boron	 10.81(2) [10.806,10.821]	m
6	C	carbon	 12.011(2)  [12.0096,12.0116]
7	N	nitrogen	 14.007(1)  [14.00643,14.00728]	m
8	O	oxygen	 15.999(1)  [15.99903,15.99977]	m
9	F	fluorine	  18.998403162(5)
10	Ne	neon	  20.1797(6)	g m
11	Na	sodium	  22.98976928(2)
12	Mg	magnesium	 24.305(2)   [24.304,24.307]
13	Al	aluminium	  26.9815384(3)
14	Si	silicon	 28.085(1)  [28.084,28.086]
15	P	phosphorus	  30.973761998(5)
16	S	sulfur	 32.06(2)   [32.059,32.076]
17	Cl	chlorine	 35.45(1)    [35.446,35.457]	m
18	Ar	argon	 39.95(16)  [39.792,39.963]
19	K	potassium	  39.0983(1)
20	Ca	calcium	  40.078(4)	g
21	Sc	scandium	  44.955907(4)
22	Ti	titanium	  47.867(1)
23	V	vanadium	  50.9415(1)
24	Cr	chromium	  51.9961(6)
25	Mn	manganese	  54.938043(2)
26	Fe	iron	  55.845(2)
27	Co	cobalt	  58.933194(3)
28	Ni	nickel	  58.6934(4)	r
29	Cu	copper	  63.546(3)	r
30	Zn	zinc	  65.38(2)	r
31	Ga	gallium	  69.723(1)
32	Ge	germanium	  72.630(8)
33	As	arsenic	  74.921595(6)
34	Se	selenium	  78.971(8)	r
35	Br	bromine	 79.904(3) [79.901,79.907]
36	Kr	krypton	  83.798(2)	g m
37	Rb	rubidium	  85.4678(3)	g
38	Sr	strontium	  87.62(1)	g r
39	Y	yttrium	  88.905838(2)
40	Zr	zirconium	  91.224(2)	g
41	Nb	niobium	  92.90637(1)
42	Mo	molybdenum	  95.95(1)	g
44	Ru	ruthenium	 101.07(2)	g
45	Rh	rhodium	 102.90549(2)
46	Pd	palladium	 106.42(1)	g
47	Ag	silver	 107.8682(2)	g
48	Cd	cadmium	 112.414(4)	g
49	In	indium	 114.818(1)
50	Sn	tin	 118.710(7)	g
51	Sb	antimony	 121.760(1)	g
52	Te	tellurium	 127.60(3)	g
53	I	iodine	 126.90447(3)
54	Xe	xenon	 131.293(6)	g m
55	Cs	caesium	 132.90545196(6)
56	Ba	barium	 137.327(7)
57	La	lanthanum	 138.90547(7)	g
58	Ce	cerium	 140.116(1)	g
59	Pr	praseodymium	 140.90766(1)
60	Nd	neodymium	 144.242(3)	g
62	Sm	samarium	 150.36(2)	g
63	Eu	europium	 151.964(1)	g
64	Gd	gadolinium	 157.25(3)	g
65	Tb	terbium	 158.925354(7)
66	Dy	dysprosium	 162.500(1)	g
67	Ho	holmium	 164.930329(5)
68	Er	erbium	 167.259(3)	g
69	Tm	thulium	 168.934219(5)
70	Yb	ytterbium	 173.045(10)	g
71	Lu	lutetium	 174.9668(1)	g
72	Hf	hafnium	 178.486(6)	g
73	Ta	tantalum	 180.94788(2)
74	W	tungsten	 183.84(1)
75	Re	rhenium	 186.207(1)
76	Os	osmium	 190.23(3)	g
77	Ir	iridium	 192.217(2)
78	Pt	platinum	 195.084(9)
79	Au	gold	 196.966570(4)
80	Hg	mercury	 200.592(3)
81	Tl	thallium	204.38(1)   [204.382,204.385]
82	Pb	lead	207.2(1.1)  [206.14,207.94]
83	Bi	bismuth	 208.98040(1)
90	Th	thorium	 232.0377(4)	g
91	Pa	protactinium	 231.03588(1)
92	U	uranium	 238.02891(3)	g m\
"""


# From https://www.ciaaw.org/isotopic-abundances.htm (2023-07-06)
# CIAAW. Isotopic compositions of the elements 2021. Available online at www.ciaaw.org.

# Nominal values for lead are from Meija (2013) since the natural variation
# given in the 2021 tables are completely unusable. Maybe better to use NaN?
# Similarly nitrogen and argon were chosen to match air rather than average
# terrestrial abundance.
# All other values use midpoint of the range, scaled so that the sum of the
# isotopes equals 100%.

#Z 	El 	element
#  isotope 	Representative isotopic composition 	Notes
isotope_abundance = """\
1	H	hydrogen
    1	[0.99972,0.99999]	m
    2	[0.00001,0.00028]
2	He	helium
    3	 0.000002(2)	g r
    4	 0.999998(2)
3	Li	lithium
    6	[0.019,0.078]	m
    7	[0.922,0.981]
4	Be	beryllium
    9	 1
5	B	boron
    10	[0.189,0.204]	m
    11	[0.796,0.811]
6	C	carbon
    12	[0.9884,0.9904]
    13	[0.0096,0.0116]
7	N	nitrogen
    14	0.996337(4) [0.99578,0.99663]
    15	0.003663(4) [0.00337,0.00422]
8	O	oxygen
    16	[0.99738,0.99776]
    17	[0.000367,0.000400]
    18	[0.00187,0.00222]
9	F	fluorine
    19	 1
10	Ne	neon
    20	 0.9048(3)	g m
    21	 0.0027(1)
    22	 0.0925(3)
11	Na	sodium
    23	 1
12	Mg	magnesium
    24	[0.7888,0.7905]
    25	[0.09988,0.10034]
    26	[0.1096,0.1109]
13	Al	aluminium
    27	 1
14	Si	silicon
    28	[0.92191,0.92318]
    29	[0.04645,0.04699]
    30	[0.03037,0.03110]
15	P	phosphorus
    31	 1
16	S	sulfur
    32	[0.9441,0.9529]
    33	[0.00729,0.00797]
    34	[0.0396,0.0477]
    36	[0.000129,0.000187]
17	Cl	chlorine
    35	[0.755,0.761]	m
    37	[0.239,0.245]
18	Ar	argon
    36	0.0033361(35) [0.0000,0.0207]	g r
    38	0.0006289(12) [0.000,0.043]
    40	0.9960350(42) [0.936,1.000]
19	K	potassium
    39	 0.932581(44)
    40	 0.000117(1)
    41	 0.067302(44)
20	Ca	calcium
    40	 0.96941(156)	g
    42	 0.00647(23)
    43	 0.00135(10)
    44	 0.02086(110)
    46	 0.00004(3)
    48	 0.00187(21)
21	Sc	scandium
    45	 1
22	Ti	titanium
    46	 0.0825(3)
    47	 0.0744(2)
    48	 0.7372(3)
    49	 0.0541(2)
    50	 0.0518(2)
23	V	vanadium
    50	 0.00250(10)
    51	 0.99750(10)
24	Cr	chromium
    50	 0.04345(13)
    52	 0.83789(18)
    53	 0.09501(17)
    54	 0.02365(7)
25	Mn	manganese
    55	 1
26	Fe	iron
    54	 0.05845(105)
    56	 0.91754(106)
    57	 0.02119(29)
    58	 0.00282(12)
27	Co	cobalt
    59	 1
28	Ni	nickel
    58	 0.680769(190)	r
    60	 0.262231(150)
    61	 0.011399(13)
    62	 0.036345(40)
    64	 0.009256(19)
29	Cu	copper
    63	 0.6915(15)	r
    65	 0.3085(15)
30	Zn	zinc
    64	 0.4917(75)	r
    66	 0.2773(98)
    67	 0.0404(16)
    68	 0.1845(63)
    70	 0.0061(10)
31	Ga	gallium
    69	 0.60108(50)
    71	 0.39892(50)
32	Ge	germanium
    70	 0.2052(19)
    72	 0.2745(15)
    73	 0.0776(8)
    74	 0.3652(12)
    76	 0.0775(12)
33	As	arsenic
    75	 1
34	Se	selenium
    74	 0.0086(3)	r
    76	 0.0923(7)
    77	 0.0760(7)
    78	 0.2369(22)
    80	 0.4980(36)
    82	 0.0882(15)
35	Br	bromine
    79	[0.505,0.508]
    81	[0.492,0.495]
36	Kr	krypton
    78	 0.00355(3)	g m
    80	 0.02286(10)
    82	 0.11593(31)
    83	 0.11500(19)
    84	 0.56987(15)
    86	 0.17279(41)
37	Rb	rubidium
    85	 0.7217(2)	g
    87	 0.2783(2)
38	Sr	strontium
    84	 0.0056(2)	g r
    86	 0.0986(20)
    87	 0.0700(20)
    88	 0.8258(35)
39	Y	yttrium
    89	 1
40	Zr	zirconium
    90	 0.5145(4)	g
    91	 0.1122(5)
    92	 0.1715(3)
    94	 0.1738(4)
    96	 0.0280(2)
41	Nb	niobium
    93	 1
42	Mo	molybdenum
    92	 0.14649(106)	g
    94	 0.09187(33)
    95	 0.15873(30)
    96	 0.16673(8)
    97	 0.09582(15)
    98	 0.24292(80)
    100	 0.09744(65)
44	Ru	ruthenium
    96	 0.0554(14)	g
    98	 0.0187(3)
    99	 0.1276(14)
    100	 0.1260(7)
    101	 0.1706(2)
    102	 0.3155(14)
    104	 0.1862(27)
45	Rh	rhodium
    103	 1
46	Pd	palladium
    102	 0.0102(1)	g
    104	 0.1114(8)
    105	 0.2233(8)
    106	 0.2733(3)
    108	 0.2646(9)
    110	 0.1172(9)
47	Ag	silver
    107	 0.51839(8)	g
    109	 0.48161(8)
48	Cd	cadmium
    106	 0.01245(22)	g
    108	 0.00888(11)
    110	 0.12470(61)
    111	 0.12795(12)
    112	 0.24109(7)
    113	 0.12227(7)
    114	 0.28754(81)
    116	 0.07512(54)
49	In	indium
    113	 0.04281(52)
    115	 0.95719(52)
50	Sn	tin
    112	 0.0097(1)	g
    114	 0.0066(1)
    115	 0.0034(1)
    116	 0.1454(9)
    117	 0.0768(7)
    118	 0.2422(9)
    119	 0.0859(4)
    120	 0.3258(9)
    122	 0.0463(3)
    124	 0.0579(5)
51	Sb	antimony
    121	 0.5721(5)	g
    123	 0.4279(5)
52	Te	tellurium
    120	 0.0009(1)	g
    122	 0.0255(12)
    123	 0.0089(3)
    124	 0.0474(14)
    125	 0.0707(15)
    126	 0.1884(25)
    128	 0.3174(8)
    130	 0.3408(62)
53	I	iodine
    127	 1
54	Xe	xenon
    124	 0.00095(5)	g m
    126	 0.00089(3)
    128	 0.01910(13)
    129	 0.26401(138)
    130	 0.04071(22)
    131	 0.21232(51)
    132	 0.26909(55)
    134	 0.10436(35)
    136	 0.08857(72)
55	Cs	caesium
    133	 1
56	Ba	barium
    130	 0.0011(1)
    132	 0.0010(1)
    134	 0.0242(15)
    135	 0.0659(10)
    136	 0.0785(24)
    137	 0.1123(23)
    138	 0.7170(29)
57	La	lanthanum
    138	 0.0008881(71)	g
    139	 0.9991119(71)
58	Ce	cerium
    136	 0.00185(2)	g
    138	 0.00251(2)
    140	 0.88450(51)
    142	 0.11114(51)
59	Pr	praseodymium
    141	 1
60	Nd	neodymium
    142	 0.27152(40)	g
    143	 0.12174(26)
    144	 0.23798(19)
    145	 0.08293(12)
    146	 0.17189(32)
    148	 0.05756(21)
    150	 0.05638(28)
62	Sm	samarium
    144	 0.0308(4)	g
    147	 0.1500(14)
    148	 0.1125(9)
    149	 0.1382(10)
    150	 0.0737(9)
    152	 0.2674(9)
    154	 0.2274(14)
63	Eu	europium
    151	 0.4781(6)	g
    153	 0.5219(6)
64	Gd	gadolinium
    152	 0.0020(3)	g
    154	 0.0218(2)
    155	 0.1480(9)
    156	 0.2047(3)
    157	 0.1565(4)
    158	 0.2484(8)
    160	 0.2186(3)
65	Tb	terbium
	159	 1
66	Dy	dysprosium
	156	 0.00056(3)	g
    158	 0.00095(3)
    160	 0.02329(18)
    161	 0.18889(42)
    162	 0.25475(36)
    163	 0.24896(42)
    164	 0.28260(54)
67	Ho	holmium
	165	 1
68	Er	erbium
	162	 0.00139(5)	g
    164	 0.01601(3)
    166	 0.33503(36)
    167	 0.22869(9)
    168	 0.26978(18)
    170	 0.14910(36)
69	Tm	thulium
	169	 1
70	Yb	ytterbium
	168	 0.00126(1)	g
    170	 0.03023(2)
    171	 0.14216(7)
    172	 0.21754(10)
    173	 0.16098(9)
    174	 0.31896(26)
    176	 0.12887(30)
71	Lu	lutetium
	175	 0.97401(13)	g
    176	 0.02599(13)
72	Hf	hafnium
	174	 0.00161(2)	g
    176	 0.0524(14)
    177	 0.1858(9)
    178	 0.2728(6)
    179	 0.1363(3)
    180	 0.3512(16)
73	Ta	tantalum
	180	 0.0001176(23)
    181	 0.9998824(23)
74	W	tungsten
	180	 0.0012(1)
    182	 0.2650(16)
    183	 0.1431(4)
    184	 0.3064(2)
    186	 0.2843(19)
75	Re	rhenium
	185	 0.3740(5)
    187	 0.6260(5)
76	Os	osmium
	184	 0.0002(2)	g
    186	 0.0159(64)
    187	 0.0196(17)
    188	 0.1324(27)
    189	 0.1615(23)
    190	 0.2626(20)
    192	 0.4078(32)
77	Ir	iridium
	191	 0.3723(9)
    193	 0.6277(9)
78	Pt	platinum
	190	 0.00012(2)
    192	 0.00782(24)
    194	 0.32864(410)
    195	 0.33775(240)
    196	 0.25211(340)
    198	 0.07356(130)
79	Au	gold
	197	 1
80	Hg	mercury
	196	 0.0015(1)
    198	 0.1004(3)
    199	 0.1694(12)
    200	 0.2314(9)
    201	 0.1317(9)
    202	 0.2974(13)
    204	 0.0682(4)
81	Tl	thallium
	203	[0.2944,0.2959]
    205	[0.7041,0.7056]
82	Pb	lead
	204	0.014(6) [0.0000,0.0158]
    206	0.241(30) [0.0190,0.8673]
    207	0.221(50) [0.0035,0.2351]
    208	0.524(70) [0.0338,0.9775]
83	Bi	bismuth
	209	 1
90	Th	thorium
	230	 0.0002(2)	g
    232	 0.9998(2)
91	Pa	protactinium
	231	 1
92	U	uranium
	234	 0.000054(5)	g m
    235	 0.007204(6)
    238	 0.992742(10)\
"""


# Table of isotope masses from Wang(2021). The mass uncertainties have been rounded to one
# or two digits, but trailing zeros remain. Hashes `#` after mass(unc.) indicate
# values which are not solely derived from experiment, see the paper for details.
# Atomic weights and isotope ratios come from Coursey(?) downloaded c. 2010.
# These are ignored, and will be replaced by the IUPAC CIAAW values from 2021.
# Meng Wang et al. (2021) Chinese Phys. C 45 030003
#    DOI:10.1088/1674-1137/abddaf
# Coursey. J. S., Schwab. D. J., and Dragoset. R. A., NIST,
#   Physics Laboratory, Office of Electronic Commerce in Scientific
#   and Engineering Data.
#
# Column layout:
#     z-el-iso,isotope mass(unc)#?,abundance(unc),element mass(unc)
isotope_mass = """\
1-H-1,1.0078250319000(100),99.9885(70),1.00794(7)
1-H-2,2.0141017778400(200),0.0115(70),1.00794(7)
1-H-3,3.0160492813200(800),,1.00794(7)
1-H-4,4.026430(110),,1.00794(7)
1-H-5,5.035310(100),,1.00794(7)
1-H-6,6.044960(270),,1.00794(7)
2-He-3,3.0160293219700(600),0.000137(3),4.002602(2)
2-He-4,4.002603254130(160),99.999863(3),4.002602(2)
2-He-5,5.0120570(210),,4.002602(2)
2-He-6,6.0188858900(600),,4.002602(2)
2-He-7,7.02799100(800),,4.002602(2)
2-He-8,8.033934390(100),,4.002602(2)
2-He-9,9.0439500(500),,4.002602(2)
2-He-10,10.052820(100),,4.002602(2)
3-Li-4,4.027190(230),,6.941(2)
3-Li-5,5.0125400(500),,6.941(2)
3-Li-6,6.01512288740(150),7.59(4),6.941(2)
3-Li-7,7.01600343400(400),92.41(4),6.941(2)
3-Li-8,8.0224862400(500),,6.941(2)
3-Li-9,9.026790190(200),,6.941(2)
3-Li-10,10.0354830(140),,6.941(2)
3-Li-11,11.043723600(700),,6.941(2)
3-Li-12,12.0526100(300),,6.941(2)
4-Be-5,5.03987(215)#,,9.012182(3)
4-Be-6,6.01972600(600),,9.012182(3)
4-Be-7,7.0169287100(800),,9.012182(3)
4-Be-8,8.0053051000(400),,9.012182(3)
4-Be-9,9.0121830600(800),100,9.012182(3)
4-Be-10,10.0135346900(900),,9.012182(3)
4-Be-11,11.021661080(260),,9.012182(3)
4-Be-12,12.02692210(200),,9.012182(3)
4-Be-13,13.0361350(110),,9.012182(3)
4-Be-14,14.042890(140),,9.012182(3)
5-B-7,7.0297120(270),,10.811(7)
5-B-8,8.02460730(110),,10.811(7)
5-B-9,9.01332960(100),,10.811(7)
5-B-10,10.0129368620(160),19.9(7),10.811(7)
5-B-11,11.0093051670(130),80.1(7),10.811(7)
5-B-12,12.01435260(140),,10.811(7)
5-B-13,13.01778000(110),,10.811(7)
5-B-14,14.0254040(230),,10.811(7)
5-B-15,15.0310870(230),,10.811(7)
5-B-16,16.0398410(260),,10.811(7)
5-B-17,17.046930(220),,10.811(7)
5-B-18,18.055600(220),,10.811(7)
5-B-19,19.064170(560),,10.811(7)
6-C-8,8.0376430(200),,12.0107(8)
6-C-9,9.03103720(230),,12.0107(8)
6-C-10,10.0168532200(800),,12.0107(8)
6-C-11,11.0114326000(600),,12.0107(8)
6-C-12,12.0(0),98.93(8),12.0107(8)
6-C-13,13.003354835340(250),1.07(8),12.0107(8)
6-C-14,14.00324198900(400),,12.0107(8)
6-C-15,15.010599300(900),,12.0107(8)
6-C-16,16.01470100(400),,12.0107(8)
6-C-17,17.0225790(190),,12.0107(8)
6-C-18,18.0267500(300),,12.0107(8)
6-C-19,19.034800(110),,12.0107(8)
6-C-20,20.040260(250),,12.0107(8)
6-C-21,21.049000(640)#,,12.0107(8)
6-C-22,22.057550(250),,12.0107(8)
7-N-10,10.041650(430),,14.0067(2)
7-N-11,11.02615800(500),,14.0067(2)
7-N-12,12.01861320(110),,14.0067(2)
7-N-13,13.005738610(290),,14.0067(2)
7-N-14,14.003074004250(240),99.632(7),14.0067(2)
7-N-15,15.000108898300(600),0.368(7),14.0067(2)
7-N-16,16.00610190(250),,14.0067(2)
7-N-17,17.0084490(160),,14.0067(2)
7-N-18,18.0140780(200),,14.0067(2)
7-N-19,19.0170220(180),,14.0067(2)
7-N-20,20.0233700(800),,14.0067(2)
7-N-21,21.027090(140),,14.0067(2)
7-N-22,22.034100(220),,14.0067(2)
7-N-23,23.039420(450),,14.0067(2)
7-N-24,24.050390(430)#,,14.0067(2)
8-O-12,12.0343680(130),,15.9994(3)
8-O-13,13.0248150(100),,15.9994(3)
8-O-14,14.0085967060(270),,15.9994(3)
8-O-15,15.003065600(500),,15.9994(3)
8-O-16,15.994914619300(300),99.757(16),15.9994(3)
8-O-17,16.999131756000(700),0.038(1),15.9994(3)
8-O-18,17.999159612100(700),0.205(14),15.9994(3)
8-O-19,19.00357800(280),,15.9994(3)
8-O-20,20.004075400(900),,15.9994(3)
8-O-21,21.0086550(130),,15.9994(3)
8-O-22,22.0099700(600),,15.9994(3)
8-O-23,23.015700(130),,15.9994(3)
8-O-24,24.019860(180),,15.9994(3)
8-O-25,25.029340(180),,15.9994(3)
8-O-26,26.037210(180),,15.9994(3)
9-F-14,14.0343200(400),,18.9984032(5)
9-F-15,15.0177850(150),,18.9984032(5)
9-F-16,16.01146000(600),,18.9984032(5)
9-F-17,17.002095240(270),,18.9984032(5)
9-F-18,18.000937300(500),,18.9984032(5)
9-F-19,18.998403162100(900),100,18.9984032(5)
9-F-20,19.9999812500(300),,18.9984032(5)
9-F-21,20.99994890(190),,18.9984032(5)
9-F-22,22.0029990(130),,18.9984032(5)
9-F-23,23.0035300(400),,18.9984032(5)
9-F-24,24.008100(100),,18.9984032(5)
9-F-25,25.012170(100),,18.9984032(5)
9-F-26,26.020050(110),,18.9984032(5)
9-F-27,27.026980(130),,18.9984032(5)
9-F-28,28.035860(130),,18.9984032(5)
9-F-29,29.043100(560),,18.9984032(5)
10-Ne-16,16.0257510(220),,20.1797(6)
10-Ne-17,17.017714000(400),,20.1797(6)
10-Ne-18,18.005708700(400),,20.1797(6)
10-Ne-19,19.001880910(170),,20.1797(6)
10-Ne-20,19.99244017530(160),90.48(3),20.1797(6)
10-Ne-21,20.9938466900(400),0.27(1),20.1797(6)
10-Ne-22,21.9913851140(190),9.25(3),20.1797(6)
10-Ne-23,22.994466910(110),,20.1797(6)
10-Ne-24,23.993610600(600),,20.1797(6)
10-Ne-25,24.9978100(300),,20.1797(6)
10-Ne-26,26.0005160(200),,20.1797(6)
10-Ne-27,27.007570(100),,20.1797(6)
10-Ne-28,28.012130(140),,20.1797(6)
10-Ne-29,29.019750(160),,20.1797(6)
10-Ne-30,30.024990(270),,20.1797(6)
10-Ne-31,31.033470(290),,20.1797(6)
10-Ne-32,32.039720(540)#,,20.1797(6)
11-Na-18,18.026880(100),,22.989770(2)
11-Na-19,19.0138800(110),,22.989770(2)
11-Na-20,20.00735430(120),,22.989770(2)
11-Na-21,20.9976544600(500),,22.989770(2)
11-Na-22,21.994437550(140),,22.989770(2)
11-Na-23,22.98976928200(190),100,22.989770(2)
11-Na-24,23.9909630120(180),,22.989770(2)
11-Na-25,24.98995400(130),,22.989770(2)
11-Na-26,25.99263500(400),,22.989770(2)
11-Na-27,26.99407600(400),,22.989770(2)
11-Na-28,27.9989390(110),,22.989770(2)
11-Na-29,29.00287700(800),,22.989770(2)
11-Na-30,30.00909800(500),,22.989770(2)
11-Na-31,31.0131470(150),,22.989770(2)
11-Na-32,32.0200100(400),,22.989770(2)
11-Na-33,33.025530(480),,22.989770(2)
11-Na-34,34.034010(640),,22.989770(2)
11-Na-35,35.040610(720)#,,22.989770(2)
12-Mg-20,20.01876310(200),,24.3050(6)
12-Mg-21,21.011705800(800),,24.3050(6)
12-Mg-22,21.999570600(170),,24.3050(6)
12-Mg-23,22.9941237700(300),,24.3050(6)
12-Mg-24,23.9850416890(140),78.99(4),24.3050(6)
12-Mg-25,24.9858369700(500),10.00(1),24.3050(6)
12-Mg-26,25.9825929700(300),11.01(3),24.3050(6)
12-Mg-27,26.9843406500(500),,24.3050(6)
12-Mg-28,27.983875430(280),,24.3050(6)
12-Mg-29,28.988607200(400),,24.3050(6)
12-Mg-30,29.99046550(140),,24.3050(6)
12-Mg-31,30.99664800(300),,24.3050(6)
12-Mg-32,31.99911000(400),,24.3050(6)
12-Mg-33,33.00532790(290),,24.3050(6)
12-Mg-34,34.00893500(700),,24.3050(6)
12-Mg-35,35.016790(290),,24.3050(6)
12-Mg-36,36.021880(740),,24.3050(6)
12-Mg-37,37.030290(750),,24.3050(6)
13-Al-21,21.029080(640)#,,26.981538(2)
13-Al-22,22.019540(430)#,,26.981538(2)
13-Al-23,23.007244400(400),,26.981538(2)
13-Al-24,23.999947600(240),,26.981538(2)
13-Al-25,24.9904283100(700),,26.981538(2)
13-Al-26,25.9868918800(700),,26.981538(2)
13-Al-27,26.9815384100(500),100,26.981538(2)
13-Al-28,27.9819100100(500),,26.981538(2)
13-Al-29,28.980453200(400),,26.981538(2)
13-Al-30,29.98296920(210),,26.981538(2)
13-Al-31,30.98394980(240),,26.981538(2)
13-Al-32,31.98808400(800),,26.981538(2)
13-Al-33,32.99087800(800),,26.981538(2)
13-Al-34,33.99678190(230),,26.981538(2)
13-Al-35,34.99976000(800),,26.981538(2)
13-Al-36,36.006390(160),,26.981538(2)
13-Al-37,37.010530(190),,26.981538(2)
13-Al-38,38.017680(160)#,,26.981538(2)
13-Al-39,39.023070(320)#,,26.981538(2)
14-Si-22,22.036110(540)#,,28.0855(3)
14-Si-23,23.025710(540)#,,28.0855(3)
14-Si-24,24.0115350(210),,28.0855(3)
14-Si-25,25.0041090(110),,28.0855(3)
14-Si-26,25.992333820(120),,28.0855(3)
14-Si-27,26.986704690(120),,28.0855(3)
14-Si-28,27.976926534400(600),92.2297(7),28.0855(3)
14-Si-29,28.976494664300(600),4.6832(5),28.0855(3)
14-Si-30,29.9737701370(230),3.0872(5),28.0855(3)
14-Si-31,30.9753632000(500),,28.0855(3)
14-Si-32,31.974151500(300),,28.0855(3)
14-Si-33,32.977977000(800),,28.0855(3)
14-Si-34,33.978538000(900),,28.0855(3)
14-Si-35,34.9845500(400),,28.0855(3)
14-Si-36,35.9866500(800),,28.0855(3)
14-Si-37,36.992950(120),,28.0855(3)
14-Si-38,37.995520(110),,28.0855(3)
14-Si-39,39.002490(150),,28.0855(3)
14-Si-40,40.006080(130),,28.0855(3)
14-Si-41,41.014170(320)#,,28.0855(3)
14-Si-42,42.018080(320)#,,28.0855(3)
15-P-24,24.036520(540)#,,30.973761(2)
15-P-25,25.021680(430)#,,30.973761(2)
15-P-26,26.011780(210)#,,30.973761(2)
15-P-27,26.9992920(100),,30.973761(2)
15-P-28,27.99232650(120),,30.973761(2)
15-P-29,28.981800400(400),,30.973761(2)
15-P-30,29.9783134900(700),,30.973761(2)
15-P-31,30.973761997700(800),100,30.973761(2)
15-P-32,31.9739076400(400),,30.973761(2)
15-P-33,32.97172570(120),,30.973761(2)
15-P-34,33.973645900(900),,30.973761(2)
15-P-35,34.97331400(200),,30.973761(2)
15-P-36,35.9782600(140),,30.973761(2)
15-P-37,36.9796100(400),,30.973761(2)
15-P-38,37.9843000(800),,30.973761(2)
15-P-39,38.986290(120),,30.973761(2)
15-P-40,39.9912600(900),,30.973761(2)
15-P-41,40.994650(130),,30.973761(2)
15-P-42,42.001170(100),,30.973761(2)
15-P-43,43.005410(320)#,,30.973761(2)
15-P-44,44.011930(430)#,,30.973761(2)
15-P-45,45.017130(540)#,,30.973761(2)
15-P-46,46.024520(540)#,,30.973761(2)
16-S-26,26.029720(640)#,,32.065(5)
16-S-27,27.018780(430)#,,32.065(5)
16-S-28,28.004370(170),,32.065(5)
16-S-29,28.9966780(140),,32.065(5)
16-S-30,29.984906770(220),,32.065(5)
16-S-31,30.979557000(250),,32.065(5)
16-S-32,31.97207117350(140),94.93(31),32.065(5)
16-S-33,32.97145890860(140),0.76(2),32.065(5)
16-S-34,33.9678670100(500),4.29(28),32.065(5)
16-S-35,34.9690323200(400),,32.065(5)
16-S-36,35.967080690(200),0.02(1),32.065(5)
16-S-37,36.971125500(210),,32.065(5)
16-S-38,37.97116300(800),,32.065(5)
16-S-39,38.9751300(500),,32.065(5)
16-S-40,39.97548300(400),,32.065(5)
16-S-41,40.97959300(400),,32.065(5)
16-S-42,41.98106500(300),,32.065(5)
16-S-43,42.98690800(500),,32.065(5)
16-S-44,43.99011900(600),,32.065(5)
16-S-45,44.996410(320)#,,32.065(5)
16-S-46,46.000690(430)#,,32.065(5)
16-S-47,47.007730(430)#,,32.065(5)
16-S-48,48.013300(540)#,,32.065(5)
16-S-49,49.021890(630)#,,32.065(5)
17-Cl-28,28.030350(540)#,,35.453(2)
17-Cl-29,29.015050(200)#,,35.453(2)
17-Cl-30,30.0050180(260),,35.453(2)
17-Cl-31,30.99244800(400),,35.453(2)
17-Cl-32,31.985684600(600),,35.453(2)
17-Cl-33,32.977452000(400),,35.453(2)
17-Cl-34,33.9737624900(500),,35.453(2)
17-Cl-35,34.9688526900(400),75.78(4),35.453(2)
17-Cl-36,35.9683068200(400),,35.453(2)
17-Cl-37,36.9659025700(600),24.22(4),35.453(2)
17-Cl-38,37.968010410(110),,35.453(2)
17-Cl-39,38.96800820(190),,35.453(2)
17-Cl-40,39.9704200(300),,35.453(2)
17-Cl-41,40.9706800(700),,35.453(2)
17-Cl-42,41.9733400(600),,35.453(2)
17-Cl-43,42.9740600(700),,35.453(2)
17-Cl-44,43.9780100(900),,35.453(2)
17-Cl-45,44.980390(150),,35.453(2)
17-Cl-46,45.985250(100),,35.453(2)
17-Cl-47,46.989720(220)#,,35.453(2)
17-Cl-48,47.995410(540)#,,35.453(2)
17-Cl-49,49.000790(430)#,,35.453(2)
17-Cl-50,50.008270(430)#,,35.453(2)
17-Cl-51,51.015340(750)#,,35.453(2)
18-Ar-30,30.023690(190)#,,39.948(1)
18-Ar-31,31.012160(220)#,,39.948(1)
18-Ar-32,31.99763780(190),,39.948(1)
18-Ar-33,32.989925500(400),,39.948(1)
18-Ar-34,33.9802700900(800),,39.948(1)
18-Ar-35,34.975257700(700),,39.948(1)
18-Ar-36,35.9675451060(290),0.3365(30),39.948(1)
18-Ar-37,36.966776300(220),,39.948(1)
18-Ar-38,37.962732100(210),0.0632(5),39.948(1)
18-Ar-39,38.96431300(500),,39.948(1)
18-Ar-40,39.96238312200(230),99.6003(30),39.948(1)
18-Ar-41,40.964500600(400),,39.948(1)
18-Ar-42,41.96304600(600),,39.948(1)
18-Ar-43,42.96563600(600),,39.948(1)
18-Ar-44,43.96492380(170),,39.948(1)
18-Ar-45,44.968039700(600),,39.948(1)
18-Ar-46,45.96803920(250),,39.948(1)
18-Ar-47,46.97276710(130),,39.948(1)
18-Ar-48,47.9760010(180),,39.948(1)
18-Ar-49,48.981690(430)#,,39.948(1)
18-Ar-50,49.985800(540)#,,39.948(1)
18-Ar-51,50.993030(430)#,,39.948(1)
18-Ar-52,51.998520(640)#,,39.948(1)
18-Ar-53,53.007290(750)#,,39.948(1)
19-K-32,32.023610(430)#,,39.0983(1)
19-K-33,33.008100(220)#,,39.0983(1)
19-K-34,33.998690(210)#,,39.0983(1)
19-K-35,34.988005400(600),,39.0983(1)
19-K-36,35.981301900(300),,39.0983(1)
19-K-37,36.973375890(100),,39.0983(1)
19-K-38,37.969081110(210),,39.0983(1)
19-K-39,38.96370648500(500),93.2581(44),39.0983(1)
19-K-40,39.9639981700(600),0.0117(1),39.0983(1)
19-K-41,40.96182525600(400),6.7302(44),39.0983(1)
19-K-42,41.962402310(110),,39.0983(1)
19-K-43,42.960734700(400),,39.0983(1)
19-K-44,43.961587000(500),,39.0983(1)
19-K-45,44.960691500(600),,39.0983(1)
19-K-46,45.961981600(800),,39.0983(1)
19-K-47,46.96166160(150),,39.0983(1)
19-K-48,47.965341200(800),,39.0983(1)
19-K-49,48.968210800(900),,39.0983(1)
19-K-50,49.97238000(800),,39.0983(1)
19-K-51,50.9758290(140),,39.0983(1)
19-K-52,51.9816000(400),,39.0983(1)
19-K-53,52.986800(120),,39.0983(1)
19-K-54,53.994470(430)#,,39.0983(1)
19-K-55,55.000510(540)#,,39.0983(1)
20-Ca-34,34.015990(320)#,,40.078(4)
20-Ca-35,35.005570(220)#,,40.078(4)
20-Ca-36,35.9930700(400),,40.078(4)
20-Ca-37,36.985897800(700),,40.078(4)
20-Ca-38,37.976319220(210),,40.078(4)
20-Ca-39,38.970710800(600),,40.078(4)
20-Ca-40,39.9625908510(220),96.941(156),40.078(4)
20-Ca-41,40.962277910(150),,40.078(4)
20-Ca-42,41.958617780(160),0.647(23),40.078(4)
20-Ca-43,42.958766380(240),0.135(10),40.078(4)
20-Ca-44,43.955481500(300),2.086(110),40.078(4)
20-Ca-45,44.956186300(400),,40.078(4)
20-Ca-46,45.95368770(240),0.004(3),40.078(4)
20-Ca-47,46.95454110(240),,40.078(4)
20-Ca-48,47.9525226540(190),0.187(21),40.078(4)
20-Ca-49,48.955662630(190),,40.078(4)
20-Ca-50,49.95749920(170),,40.078(4)
20-Ca-51,50.960995700(600),,40.078(4)
20-Ca-52,51.963213600(700),,40.078(4)
20-Ca-53,52.9684500(500),,40.078(4)
20-Ca-54,53.9729900(500),,40.078(4)
20-Ca-55,54.979980(170),,40.078(4)
20-Ca-56,55.985500(270),,40.078(4)
20-Ca-57,56.992960(430)#,,40.078(4)
21-Sc-36,36.017340(320)#,,44.955910(8)
21-Sc-37,37.004060(320)#,,44.955910(8)
21-Sc-38,37.995440(220)#,,44.955910(8)
21-Sc-39,38.9847850(260),,44.955910(8)
21-Sc-40,39.97796700(300),,44.955910(8)
21-Sc-41,40.9692511600(800),,44.955910(8)
21-Sc-42,41.965516690(170),,44.955910(8)
21-Sc-43,42.96115040(200),,44.955910(8)
21-Sc-44,43.95940280(190),,44.955910(8)
21-Sc-45,44.955907100(700),100,44.955910(8)
21-Sc-46,45.955167000(700),,44.955910(8)
21-Sc-47,46.95240240(210),,44.955910(8)
21-Sc-48,47.95222300(500),,44.955910(8)
21-Sc-49,48.95001320(240),,44.955910(8)
21-Sc-50,49.95218740(270),,44.955910(8)
21-Sc-51,50.95356880(270),,44.955910(8)
21-Sc-52,51.95649600(300),,44.955910(8)
21-Sc-53,52.9583790(190),,44.955910(8)
21-Sc-54,53.9630290(150),,44.955910(8)
21-Sc-55,54.9668900(700),,44.955910(8)
21-Sc-56,55.972610(280),,44.955910(8)
21-Sc-57,56.977050(190),,44.955910(8)
21-Sc-58,57.983380(200),,44.955910(8)
21-Sc-59,58.988370(270),,44.955910(8)
22-Ti-38,38.012210(320)#,,47.867(1)
22-Ti-39,39.002680(220)#,,47.867(1)
22-Ti-40,39.9903500(700),,47.867(1)
22-Ti-41,40.9831500(300),,47.867(1)
22-Ti-42,41.973049370(290),,47.867(1)
22-Ti-43,42.96852800(600),,47.867(1)
22-Ti-44,43.959689900(800),,47.867(1)
22-Ti-45,44.958120800(900),,47.867(1)
22-Ti-46,45.952626360(100),8.25(3),47.867(1)
22-Ti-47,46.9517574900(900),7.44(2),47.867(1)
22-Ti-48,47.9479406800(800),73.72(3),47.867(1)
22-Ti-49,48.9478643900(800),5.41(2),47.867(1)
22-Ti-50,49.9447856200(900),5.18(2),47.867(1)
22-Ti-51,50.946609500(500),,47.867(1)
22-Ti-52,51.94688350(290),,47.867(1)
22-Ti-53,52.94967100(300),,47.867(1)
22-Ti-54,53.9508920(170),,47.867(1)
22-Ti-55,54.9550900(300),,47.867(1)
22-Ti-56,55.957680(110),,47.867(1)
22-Ti-57,56.963070(220),,47.867(1)
22-Ti-58,57.966810(200),,47.867(1)
22-Ti-59,58.972220(320)#,,47.867(1)
22-Ti-60,59.976280(260),,47.867(1)
22-Ti-61,60.982430(320)#,,47.867(1)
23-V-40,40.013390(320)#,,50.9415(1)
23-V-41,41.000330(220)#,,50.9415(1)
23-V-42,41.991820(210)#,,50.9415(1)
23-V-43,42.9807700(500),,50.9415(1)
23-V-44,43.97444100(800),,50.9415(1)
23-V-45,44.965768500(900),,50.9415(1)
23-V-46,45.960197390(140),,50.9415(1)
23-V-47,46.954903560(120),,50.9415(1)
23-V-48,47.95225090(100),,50.9415(1)
23-V-49,48.948510500(900),,50.9415(1)
23-V-50,49.947156680(100),0.250(4),50.9415(1)
23-V-51,50.943957660(100),99.750(4),50.9415(1)
23-V-52,51.944773640(170),,50.9415(1)
23-V-53,52.94433500(300),,50.9415(1)
23-V-54,53.9464320(120),,50.9415(1)
23-V-55,54.9472620(290),,50.9415(1)
23-V-56,55.950420(190),,50.9415(1)
23-V-57,56.9523000(900),,50.9415(1)
23-V-58,57.956600(100),,50.9415(1)
23-V-59,58.959620(150),,50.9415(1)
23-V-60,59.964480(200),,50.9415(1)
23-V-61,60.967600(250),,50.9415(1)
23-V-62,61.972930(280),,50.9415(1)
23-V-63,62.976660(370),,50.9415(1)
24-Cr-42,42.007580(320)#,,51.9961(6)
24-Cr-43,42.997890(220)#,,51.9961(6)
24-Cr-44,43.9855900(600),,51.9961(6)
24-Cr-45,44.9790500(400),,51.9961(6)
24-Cr-46,45.9683610(120),,51.9961(6)
24-Cr-47,46.96289500(600),,51.9961(6)
24-Cr-48,47.95402900(800),,51.9961(6)
24-Cr-49,48.95133370(240),,51.9961(6)
24-Cr-50,49.946042210(100),4.345(13),51.9961(6)
24-Cr-51,50.944765390(180),,51.9961(6)
24-Cr-52,51.940504710(120),83.789(18),51.9961(6)
24-Cr-53,52.940646300(120),9.501(17),51.9961(6)
24-Cr-54,53.938877360(140),2.365(7),51.9961(6)
24-Cr-55,54.940836640(250),,51.9961(6)
24-Cr-56,55.940649000(600),,51.9961(6)
24-Cr-57,56.94361210(200),,51.9961(6)
24-Cr-58,57.94418500(300),,51.9961(6)
24-Cr-59,58.948345400(700),,51.9961(6)
24-Cr-60,59.94964170(120),,51.9961(6)
24-Cr-61,60.95437810(200),,51.9961(6)
24-Cr-62,61.95614300(400),,51.9961(6)
24-Cr-63,62.9611600(800),,51.9961(6)
24-Cr-64,63.963890(320),,51.9961(6)
24-Cr-65,64.969610(220)#,,51.9961(6)
25-Mn-44,44.008010(320)#,,54.938049(9)
25-Mn-45,44.994650(320)#,,54.938049(9)
25-Mn-46,45.9866700(900),,54.938049(9)
25-Mn-47,46.9757700(300),,54.938049(9)
25-Mn-48,47.96854900(700),,54.938049(9)
25-Mn-49,48.95961340(240),,54.938049(9)
25-Mn-50,49.954238160(120),,54.938049(9)
25-Mn-51,50.948208800(300),,54.938049(9)
25-Mn-52,51.945559090(140),,54.938049(9)
25-Mn-53,52.941287500(400),,54.938049(9)
25-Mn-54,53.94035580(110),,54.938049(9)
25-Mn-55,54.938043040(280),100,54.938049(9)
25-Mn-56,55.938902800(300),,54.938049(9)
25-Mn-57,56.93828590(160),,54.938049(9)
25-Mn-58,57.94006660(290),,54.938049(9)
25-Mn-59,58.94039110(250),,54.938049(9)
25-Mn-60,59.94313660(250),,54.938049(9)
25-Mn-61,60.94445250(250),,54.938049(9)
25-Mn-62,61.94790700(700),,54.938049(9)
25-Mn-63,62.94966500(400),,54.938049(9)
25-Mn-64,63.95384900(400),,54.938049(9)
25-Mn-65,64.95602000(400),,54.938049(9)
25-Mn-66,65.9605470(120),,54.938049(9)
25-Mn-67,66.963950(220)#,,54.938049(9)
26-Fe-45,45.015470(300)#,,55.845(2)
26-Fe-46,46.001300(320)#,,55.845(2)
26-Fe-47,46.992350(540)#,,55.845(2)
26-Fe-48,47.980670(100),,55.845(2)
26-Fe-49,48.9734290(260),,55.845(2)
26-Fe-50,49.96298800(900),,55.845(2)
26-Fe-51,50.95685510(150),,55.845(2)
26-Fe-52,51.948113360(190),,55.845(2)
26-Fe-53,52.94530560(180),,55.845(2)
26-Fe-54,53.939608200(400),5.845(35),55.845(2)
26-Fe-55,54.938291200(300),,55.845(2)
26-Fe-56,55.934935540(290),91.754(36),55.845(2)
26-Fe-57,56.935391950(290),2.119(10),55.845(2)
26-Fe-58,57.933273600(300),0.282(4),55.845(2)
26-Fe-59,58.934873500(400),,55.845(2)
26-Fe-60,59.93407000(400),,55.845(2)
26-Fe-61,60.93674620(280),,55.845(2)
26-Fe-62,61.93679200(300),,55.845(2)
26-Fe-63,62.94027300(500),,55.845(2)
26-Fe-64,63.94098800(500),,55.845(2)
26-Fe-65,64.94501500(500),,55.845(2)
26-Fe-66,65.94625000(400),,55.845(2)
26-Fe-67,66.95093000(400),,55.845(2)
26-Fe-68,67.952880(210)#,,55.845(2)
26-Fe-69,68.957920(220)#,,55.845(2)
27-Co-48,48.001860(540)#,,58.933200(9)
27-Co-49,48.989500(540)#,,58.933200(9)
27-Co-50,49.981120(140),,58.933200(9)
27-Co-51,50.9706500(500),,58.933200(9)
27-Co-52,51.96313000(600),,58.933200(9)
27-Co-53,52.95420330(190),,58.933200(9)
27-Co-54,53.948459100(400),,58.933200(9)
27-Co-55,54.941996400(400),,58.933200(9)
27-Co-56,55.939838000(500),,58.933200(9)
27-Co-57,56.936289800(600),,58.933200(9)
27-Co-58,57.93575130(120),,58.933200(9)
27-Co-59,58.933193500(400),100,58.933200(9)
27-Co-60,59.933815500(400),,58.933200(9)
27-Co-61,60.932476000(900),,58.933200(9)
27-Co-62,61.9340580(200),,58.933200(9)
27-Co-63,62.9336000(200),,58.933200(9)
27-Co-64,63.9358100(210),,58.933200(9)
27-Co-65,64.93646210(220),,58.933200(9)
27-Co-66,65.9394430(150),,58.933200(9)
27-Co-67,66.94061000(700),,58.933200(9)
27-Co-68,67.94455900(400),,58.933200(9)
27-Co-69,68.9459100(900),,58.933200(9)
27-Co-70,69.9500530(120),,58.933200(9)
27-Co-71,70.952370(500),,58.933200(9)
27-Co-72,71.956740(320)#,,58.933200(9)
28-Ni-50,49.996290(540)#,,58.6934(2)
28-Ni-51,50.987490(540)#,,58.6934(2)
28-Ni-52,51.9757800(900),,58.6934(2)
28-Ni-53,52.9681900(270),,58.6934(2)
28-Ni-54,53.95783300(500),,58.6934(2)
28-Ni-55,54.951329800(800),,58.6934(2)
28-Ni-56,55.942127800(400),,58.6934(2)
28-Ni-57,56.939791400(600),,58.6934(2)
28-Ni-58,57.935341700(400),68.0769(89),58.6934(2)
28-Ni-59,58.934345400(400),,58.6934(2)
28-Ni-60,59.930785100(400),26.2231(77),58.6934(2)
28-Ni-61,60.931054800(400),1.1399(6),58.6934(2)
28-Ni-62,61.928344800(500),3.6345(17),58.6934(2)
28-Ni-63,62.929669000(500),,58.6934(2)
28-Ni-64,63.927966200(500),0.9256(9),58.6934(2)
28-Ni-65,64.930084600(500),,58.6934(2)
28-Ni-66,65.92913930(150),,58.6934(2)
28-Ni-67,66.93156900(300),,58.6934(2)
28-Ni-68,67.93186900(300),,58.6934(2)
28-Ni-69,68.93561000(400),,58.6934(2)
28-Ni-70,69.93643130(230),,58.6934(2)
28-Ni-71,70.94051900(240),,58.6934(2)
28-Ni-72,71.94178590(240),,58.6934(2)
28-Ni-73,72.94620670(260),,58.6934(2)
28-Ni-74,73.947720(220)#,,58.6934(2)
28-Ni-75,74.952510(220)#,,58.6934(2)
28-Ni-76,75.954710(320)#,,58.6934(2)
28-Ni-77,76.959900(430)#,,58.6934(2)
28-Ni-78,77.962560(430)#,,58.6934(2)
29-Cu-52,51.997980(640)#,,63.546(3)
29-Cu-53,52.985890(540)#,,63.546(3)
29-Cu-54,53.977200(430)#,,63.546(3)
29-Cu-55,54.966040(170),,63.546(3)
29-Cu-56,55.95852900(700),,63.546(3)
29-Cu-57,56.949211700(500),,63.546(3)
29-Cu-58,57.944532300(600),,63.546(3)
29-Cu-59,58.939496700(600),,63.546(3)
29-Cu-60,59.93736380(170),,63.546(3)
29-Cu-61,60.93345740(100),,63.546(3)
29-Cu-62,61.932594800(700),,63.546(3)
29-Cu-63,62.929597100(500),69.17(3),63.546(3)
29-Cu-64,63.929764000(500),,63.546(3)
29-Cu-65,64.927789500(700),30.83(3),63.546(3)
29-Cu-66,65.928868800(700),,63.546(3)
29-Cu-67,66.92772950(100),,63.546(3)
29-Cu-68,67.92961090(170),,63.546(3)
29-Cu-69,68.92942930(150),,63.546(3)
29-Cu-70,69.93239210(120),,63.546(3)
29-Cu-71,70.93267680(160),,63.546(3)
29-Cu-72,71.93582030(150),,63.546(3)
29-Cu-73,72.93667440(210),,63.546(3)
29-Cu-74,73.93987500(700),,63.546(3)
29-Cu-75,74.941523800(800),,63.546(3)
29-Cu-76,75.94526900(100),,63.546(3)
29-Cu-77,76.94754360(130),,63.546(3)
29-Cu-78,77.9519170(140),,63.546(3)
29-Cu-79,78.954470(110),,63.546(3)
29-Cu-80,79.960620(320)#,,63.546(3)
30-Zn-54,53.993880(230)#,,65.409(4)
30-Zn-55,54.984680(430)#,,65.409(4)
30-Zn-56,55.972740(430)#,,65.409(4)
30-Zn-57,56.965060(220)#,,65.409(4)
30-Zn-58,57.9545900(500),,65.409(4)
30-Zn-59,58.949311900(800),,65.409(4)
30-Zn-60,59.941841300(600),,65.409(4)
30-Zn-61,60.9395070(170),,65.409(4)
30-Zn-62,61.934333400(700),,65.409(4)
30-Zn-63,62.93321110(170),,65.409(4)
30-Zn-64,63.929141800(700),48.63(60),65.409(4)
30-Zn-65,64.929240500(700),,65.409(4)
30-Zn-66,65.926033600(800),27.90(27),65.409(4)
30-Zn-67,66.927127400(800),4.10(13),65.409(4)
30-Zn-68,67.924844200(800),18.75(51),65.409(4)
30-Zn-69,68.926550400(900),,65.409(4)
30-Zn-70,69.92531920(210),0.62(3),65.409(4)
30-Zn-71,70.92771960(280),,65.409(4)
30-Zn-72,71.92684280(230),,65.409(4)
30-Zn-73,72.92958260(200),,65.409(4)
30-Zn-74,73.92940730(270),,65.409(4)
30-Zn-75,74.93284020(210),,65.409(4)
30-Zn-76,75.93311500(160),,65.409(4)
30-Zn-77,76.93688720(210),,65.409(4)
30-Zn-78,77.93828920(210),,65.409(4)
30-Zn-79,78.94263810(240),,65.409(4)
30-Zn-80,79.94455290(280),,65.409(4)
30-Zn-81,80.95040300(500),,65.409(4)
30-Zn-82,81.95457400(300),,65.409(4)
31-Ga-56,55.995880(540)#,,69.723(1)
31-Ga-57,56.983460(430)#,,69.723(1)
31-Ga-58,57.974730(320)#,,69.723(1)
31-Ga-59,58.963760(180)#,,69.723(1)
31-Ga-60,59.957500(220)#,,69.723(1)
31-Ga-61,60.9494000(400),,69.723(1)
31-Ga-62,61.944189600(700),,69.723(1)
31-Ga-63,62.93929420(140),,69.723(1)
31-Ga-64,63.93684040(150),,69.723(1)
31-Ga-65,64.932734400(800),,69.723(1)
31-Ga-66,65.93158980(120),,69.723(1)
31-Ga-67,66.92820230(130),,69.723(1)
31-Ga-68,67.92798020(150),,69.723(1)
31-Ga-69,68.92557350(130),60.108(9),69.723(1)
31-Ga-70,69.92602190(130),,69.723(1)
31-Ga-71,70.924702600(900),39.892(9),69.723(1)
31-Ga-72,71.926367500(900),,69.723(1)
31-Ga-73,72.92517470(180),,69.723(1)
31-Ga-74,73.92694600(300),,69.723(1)
31-Ga-75,74.926504500(700),,69.723(1)
31-Ga-76,75.92882760(210),,69.723(1)
31-Ga-77,76.92915430(260),,69.723(1)
31-Ga-78,77.93161090(110),,69.723(1)
31-Ga-79,78.93285160(130),,69.723(1)
31-Ga-80,79.93642100(300),,69.723(1)
31-Ga-81,80.93813400(400),,69.723(1)
31-Ga-82,81.94317650(260),,69.723(1)
31-Ga-83,82.94712030(280),,69.723(1)
31-Ga-84,83.9526600(300),,69.723(1)
32-Ge-58,57.991860(540)#,,72.64(1)
32-Ge-59,58.982430(430)#,,72.64(1)
32-Ge-60,59.970450(320)#,,72.64(1)
32-Ge-61,60.963730(320)#,,72.64(1)
32-Ge-62,61.954760(150)#,,72.64(1)
32-Ge-63,62.9496300(400),,72.64(1)
32-Ge-64,63.94169000(400),,72.64(1)
32-Ge-65,64.93936810(230),,72.64(1)
32-Ge-66,65.93386210(260),,72.64(1)
32-Ge-67,66.93271700(500),,72.64(1)
32-Ge-68,67.92809530(200),,72.64(1)
32-Ge-69,68.92796450(140),,72.64(1)
32-Ge-70,69.924248500(900),20.84(87),72.64(1)
32-Ge-71,70.924952100(900),,72.64(1)
32-Ge-72,71.9220758200(800),27.54(34),72.64(1)
32-Ge-73,72.9234589500(600),7.73(5),72.64(1)
32-Ge-74,73.9211777610(130),36.28(73),72.64(1)
32-Ge-75,74.9228583700(600),,72.64(1)
32-Ge-76,75.9214027250(190),7.61(38),72.64(1)
32-Ge-77,76.9235498400(600),,72.64(1)
32-Ge-78,77.92285300(400),,72.64(1)
32-Ge-79,78.9253600(400),,72.64(1)
32-Ge-80,79.92535080(220),,72.64(1)
32-Ge-81,80.92883290(220),,72.64(1)
32-Ge-82,81.92977400(240),,72.64(1)
32-Ge-83,82.93453910(260),,72.64(1)
32-Ge-84,83.93757500(300),,72.64(1)
32-Ge-85,84.94297000(400),,72.64(1)
32-Ge-86,85.946970(470),,72.64(1)
33-As-60,59.993950(430)#,,74.92160(2)
33-As-61,60.981540(320)#,,74.92160(2)
33-As-62,61.973780(320)#,,74.92160(2)
33-As-63,62.964040(220)#,,74.92160(2)
33-As-64,63.957560(220)#,,74.92160(2)
33-As-65,64.9496100(900),,74.92160(2)
33-As-66,65.94414900(600),,74.92160(2)
33-As-67,66.939251100(500),,74.92160(2)
33-As-68,67.93677410(200),,74.92160(2)
33-As-69,68.9322500(300),,74.92160(2)
33-As-70,69.93093460(150),,74.92160(2)
33-As-71,70.92711400(400),,74.92160(2)
33-As-72,71.92675200(400),,74.92160(2)
33-As-73,72.92382900(400),,74.92160(2)
33-As-74,73.92392860(180),,74.92160(2)
33-As-75,74.921594600(900),100,74.92160(2)
33-As-76,75.92239200(100),,74.92160(2)
33-As-77,76.92064760(180),,74.92160(2)
33-As-78,77.9218280(100),,74.92160(2)
33-As-79,78.92094800(600),,74.92160(2)
33-As-80,79.92247400(400),,74.92160(2)
33-As-81,80.92213230(280),,74.92160(2)
33-As-82,81.92473900(400),,74.92160(2)
33-As-83,82.92520700(300),,74.92160(2)
33-As-84,83.92930300(300),,74.92160(2)
33-As-85,84.93216400(300),,74.92160(2)
33-As-86,85.93670200(400),,74.92160(2)
33-As-87,86.94029200(300),,74.92160(2)
33-As-88,87.945840(220)#,,74.92160(2)
33-As-89,88.950050(320)#,,74.92160(2)
34-Se-65,64.964550(320)#,,78.96(3)
34-Se-66,65.955280(220)#,,78.96(3)
34-Se-67,66.9499900(700),,78.96(3)
34-Se-68,67.941825200(500),,78.96(3)
34-Se-69,68.93941480(160),,78.96(3)
34-Se-70,69.93351550(170),,78.96(3)
34-Se-71,70.93220900(300),,78.96(3)
34-Se-72,71.92714050(210),,78.96(3)
34-Se-73,72.92675500(800),,78.96(3)
34-Se-74,73.9224759340(160),0.89(4),78.96(3)
34-Se-75,74.9225228700(800),,78.96(3)
34-Se-76,75.9192137030(170),9.37(29),78.96(3)
34-Se-77,76.9199141500(700),7.63(16),78.96(3)
34-Se-78,77.917309240(190),23.77(28),78.96(3)
34-Se-79,78.918499250(240),,78.96(3)
34-Se-80,79.91652180(100),49.61(41),78.96(3)
34-Se-81,80.91799300(100),,78.96(3)
34-Se-82,81.916699500(500),8.73(22),78.96(3)
34-Se-83,82.91911900(300),,78.96(3)
34-Se-84,83.91846680(210),,78.96(3)
34-Se-85,84.92226080(280),,78.96(3)
34-Se-86,85.92431170(270),,78.96(3)
34-Se-87,86.92868860(240),,78.96(3)
34-Se-88,87.93141700(400),,78.96(3)
34-Se-89,88.93666900(400),,78.96(3)
34-Se-90,89.940100(350),,78.96(3)
34-Se-91,90.945700(470),,78.96(3)
34-Se-92,91.949840(430)#,,78.96(3)
35-Br-67,66.965080(320)#,,79.904(1)
35-Br-68,67.958360(280)#,,79.904(1)
35-Br-69,68.9503400(500),,79.904(1)
35-Br-70,69.9447920(160),,79.904(1)
35-Br-71,70.93934200(600),,79.904(1)
35-Br-72,71.93659460(110),,79.904(1)
35-Br-73,72.93167300(700),,79.904(1)
35-Br-74,73.92991000(600),,79.904(1)
35-Br-75,74.92581100(500),,79.904(1)
35-Br-76,75.9245420(100),,79.904(1)
35-Br-77,76.92137900(300),,79.904(1)
35-Br-78,77.92114600(400),,79.904(1)
35-Br-79,78.91833760(110),50.69(7),79.904(1)
35-Br-80,79.91852980(110),,79.904(1)
35-Br-81,80.91628820(100),49.31(7),79.904(1)
35-Br-82,81.91680180(100),,79.904(1)
35-Br-83,82.91517500(400),,79.904(1)
35-Br-84,83.9164960(280),,79.904(1)
35-Br-85,84.91564600(300),,79.904(1)
35-Br-86,85.91880500(300),,79.904(1)
35-Br-87,86.92067400(300),,79.904(1)
35-Br-88,87.92408300(300),,79.904(1)
35-Br-89,88.92670500(400),,79.904(1)
35-Br-90,89.93129300(400),,79.904(1)
35-Br-91,90.93439900(400),,79.904(1)
35-Br-92,91.93963200(700),,79.904(1)
35-Br-93,92.943220(460),,79.904(1)
35-Br-94,93.948850(220)#,,79.904(1)
36-Kr-69,68.965500(320)#,,83.798(2)
36-Kr-70,69.955880(220)#,,83.798(2)
36-Kr-71,70.950270(140),,83.798(2)
36-Kr-72,71.94209200(900),,83.798(2)
36-Kr-73,72.93928900(700),,83.798(2)
36-Kr-74,73.93308400(220),,83.798(2)
36-Kr-75,74.93094600(900),,83.798(2)
36-Kr-76,75.92591100(400),,83.798(2)
36-Kr-77,76.92467000(210),,83.798(2)
36-Kr-78,77.920366300(300),0.35(1),83.798(2)
36-Kr-79,78.92008300(400),,83.798(2)
36-Kr-80,79.916377900(700),2.28(6),83.798(2)
36-Kr-81,80.91658970(120),,83.798(2)
36-Kr-82,81.91348115400(600),11.58(14),83.798(2)
36-Kr-83,82.9141265170(100),11.49(6),83.798(2)
36-Kr-84,83.91149772700(400),57.00(4),83.798(2)
36-Kr-85,84.91252730(210),,83.798(2)
36-Kr-86,85.91061062500(400),17.30(22),83.798(2)
36-Kr-87,86.913354760(260),,83.798(2)
36-Kr-88,87.91444790(280),,83.798(2)
36-Kr-89,88.91783540(230),,83.798(2)
36-Kr-90,89.91952790(200),,83.798(2)
36-Kr-91,90.92380630(240),,83.798(2)
36-Kr-92,91.92617310(290),,83.798(2)
36-Kr-93,92.93114720(270),,83.798(2)
36-Kr-94,93.9341400(130),,83.798(2)
36-Kr-95,94.9397110(200),,83.798(2)
36-Kr-96,95.9430140(210),,83.798(2)
36-Kr-97,96.949090(140),,83.798(2)
37-Rb-71,70.965340(430)#,,85.4678(3)
37-Rb-72,71.958850(540)#,,85.4678(3)
37-Rb-73,72.9506000(400),,85.4678(3)
37-Rb-74,73.94426600(300),,85.4678(3)
37-Rb-75,74.93857320(130),,85.4678(3)
37-Rb-76,75.93507300(100),,85.4678(3)
37-Rb-77,76.93040160(140),,85.4678(3)
37-Rb-78,77.92814200(300),,85.4678(3)
37-Rb-79,78.92399010(210),,85.4678(3)
37-Rb-80,79.92251640(200),,85.4678(3)
37-Rb-81,80.91899400(500),,85.4678(3)
37-Rb-82,81.91820900(300),,85.4678(3)
37-Rb-83,82.91511420(250),,85.4678(3)
37-Rb-84,83.91437520(240),,85.4678(3)
37-Rb-85,84.91178973600(500),72.17(2),85.4678(3)
37-Rb-86,85.911167440(210),,85.4678(3)
37-Rb-87,86.90918052900(600),27.83(2),85.4678(3)
37-Rb-88,87.911315590(170),,85.4678(3)
37-Rb-89,88.91227800(600),,85.4678(3)
37-Rb-90,89.91479800(700),,85.4678(3)
37-Rb-91,90.91653700(800),,85.4678(3)
37-Rb-92,91.91972800(700),,85.4678(3)
37-Rb-93,92.92203900(800),,85.4678(3)
37-Rb-94,93.92639480(220),,85.4678(3)
37-Rb-95,94.9292640(220),,85.4678(3)
37-Rb-96,95.93413300(400),,85.4678(3)
37-Rb-97,96.93717710(210),,85.4678(3)
37-Rb-98,97.9416320(170),,85.4678(3)
37-Rb-99,98.94511900(400),,85.4678(3)
37-Rb-100,99.9503320(140),,85.4678(3)
37-Rb-101,100.9543020(220),,85.4678(3)
37-Rb-102,101.9600100(900),,85.4678(3)
38-Sr-73,72.965700(430)#,,87.62(1)
38-Sr-74,73.956170(110)#,,87.62(1)
38-Sr-75,74.949950(240),,87.62(1)
38-Sr-76,75.9417600(400),,87.62(1)
38-Sr-77,76.93794500(900),,87.62(1)
38-Sr-78,77.93218000(800),,87.62(1)
38-Sr-79,78.92970500(800),,87.62(1)
38-Sr-80,79.92451800(400),,87.62(1)
38-Sr-81,80.92321100(300),,87.62(1)
38-Sr-82,81.91840000(600),,87.62(1)
38-Sr-83,82.91755400(700),,87.62(1)
38-Sr-84,83.91341910(130),0.56(1),87.62(1)
38-Sr-85,84.91293200(300),,87.62(1)
38-Sr-86,85.90926072500(600),9.86(1),87.62(1)
38-Sr-87,86.90887749500(600),7.00(1),87.62(1)
38-Sr-88,87.90561225400(600),82.58(1),87.62(1)
38-Sr-89,88.907450810(100),,87.62(1)
38-Sr-90,89.90772790(160),,87.62(1)
38-Sr-91,90.91019600(600),,87.62(1)
38-Sr-92,91.91103800(400),,87.62(1)
38-Sr-93,92.91402400(800),,87.62(1)
38-Sr-94,93.91535560(180),,87.62(1)
38-Sr-95,94.91935800(600),,87.62(1)
38-Sr-96,95.92171900(900),,87.62(1)
38-Sr-97,96.92637600(400),,87.62(1)
38-Sr-98,97.92869300(300),,87.62(1)
38-Sr-99,98.93288400(500),,87.62(1)
38-Sr-100,99.93578300(700),,87.62(1)
38-Sr-101,100.94060600(900),,87.62(1)
38-Sr-102,101.9440000(700),,87.62(1)
38-Sr-103,102.949240(220)#,,87.62(1)
38-Sr-104,103.953020(320)#,,87.62(1)
39-Y-77,76.950150(220)#,,88.90585(2)
39-Y-78,77.943990(320)#,,88.90585(2)
39-Y-79,78.9379500(900),,88.90585(2)
39-Y-80,79.93435500(700),,88.90585(2)
39-Y-81,80.92945400(600),,88.90585(2)
39-Y-82,81.92693000(600),,88.90585(2)
39-Y-83,82.9224840(200),,88.90585(2)
39-Y-84,83.92067100(500),,88.90585(2)
39-Y-85,84.9164330(200),,88.90585(2)
39-Y-86,85.9148860(150),,88.90585(2)
39-Y-87,86.91087610(120),,88.90585(2)
39-Y-88,87.90950130(160),,88.90585(2)
39-Y-89,88.905838200(400),100,88.90585(2)
39-Y-90,89.907141700(400),,88.90585(2)
39-Y-91,90.90729800(200),,88.90585(2)
39-Y-92,91.9089460(100),,88.90585(2)
39-Y-93,92.9095780(110),,88.90585(2)
39-Y-94,93.91159200(700),,88.90585(2)
39-Y-95,94.91282000(700),,88.90585(2)
39-Y-96,95.91590900(700),,88.90585(2)
39-Y-97,96.91828700(700),,88.90585(2)
39-Y-98,97.92239500(900),,88.90585(2)
39-Y-99,98.92416100(700),,88.90585(2)
39-Y-100,99.9277280(120),,88.90585(2)
39-Y-101,100.93016100(800),,88.90585(2)
39-Y-102,101.93432800(400),,88.90585(2)
39-Y-103,102.9372440(120),,88.90585(2)
39-Y-104,103.941940(220)#,,88.90585(2)
39-Y-105,104.945710(430)#,,88.90585(2)
39-Y-106,105.950840(540)#,,88.90585(2)
40-Zr-79,78.949790(320)#,,91.224(2)
40-Zr-80,79.941210(320)#,,91.224(2)
40-Zr-81,80.938250(100),,91.224(2)
40-Zr-82,81.93170750(170),,91.224(2)
40-Zr-83,82.92924100(700),,91.224(2)
40-Zr-84,83.92332600(600),,91.224(2)
40-Zr-85,84.92144300(700),,91.224(2)
40-Zr-86,85.91629700(400),,91.224(2)
40-Zr-87,86.91481700(400),,91.224(2)
40-Zr-88,87.91022100(600),,91.224(2)
40-Zr-89,88.90887980(300),,91.224(2)
40-Zr-90,89.904698760(130),51.45(40),91.224(2)
40-Zr-91,90.905640210(100),11.22(5),91.224(2)
40-Zr-92,91.905035340(100),17.15(8),91.224(2)
40-Zr-93,92.906470700(500),,91.224(2)
40-Zr-94,93.906312520(180),17.38(28),91.224(2)
40-Zr-95,94.908040300(900),,91.224(2)
40-Zr-96,95.908277620(120),2.80(9),91.224(2)
40-Zr-97,96.910963800(130),,91.224(2)
40-Zr-98,97.91274000(900),,91.224(2)
40-Zr-99,98.9166750(110),,91.224(2)
40-Zr-100,99.91801000(900),,91.224(2)
40-Zr-101,100.92145800(900),,91.224(2)
40-Zr-102,101.92315400(900),,91.224(2)
40-Zr-103,102.9272040(100),,91.224(2)
40-Zr-104,103.9294490(100),,91.224(2)
40-Zr-105,104.9340220(130),,91.224(2)
40-Zr-106,105.936930(220)#,,91.224(2)
40-Zr-107,106.942010(320)#,,91.224(2)
40-Zr-108,107.945300(430)#,,91.224(2)
41-Nb-81,80.950230(430)#,,92.90638(2)
41-Nb-82,81.944380(320)#,,92.90638(2)
41-Nb-83,82.938150(170),,92.90638(2)
41-Nb-84,83.934305700(400),,92.90638(2)
41-Nb-85,84.92884600(400),,92.90638(2)
41-Nb-86,85.92578200(600),,92.90638(2)
41-Nb-87,86.92069200(700),,92.90638(2)
41-Nb-88,87.9182300(600),,92.90638(2)
41-Nb-89,88.9134450(250),,92.90638(2)
41-Nb-90,89.91125900(400),,92.90638(2)
41-Nb-91,90.90699000(300),,92.90638(2)
41-Nb-92,91.90718860(190),,92.90638(2)
41-Nb-93,92.90637320(160),100,92.90638(2)
41-Nb-94,93.90727900(160),,92.90638(2)
41-Nb-95,94.906831100(500),,92.90638(2)
41-Nb-96,95.908101590(160),,92.90638(2)
41-Nb-97,96.90810200(500),,92.90638(2)
41-Nb-98,97.91033300(500),,92.90638(2)
41-Nb-99,98.9116090(130),,92.90638(2)
41-Nb-100,99.91434100(900),,92.90638(2)
41-Nb-101,100.91530700(400),,92.90638(2)
41-Nb-102,101.91809040(270),,92.90638(2)
41-Nb-103,102.91945300(400),,92.90638(2)
41-Nb-104,103.92290770(190),,92.90638(2)
41-Nb-105,104.92494300(400),,92.90638(2)
41-Nb-106,105.92892850(150),,92.90638(2)
41-Nb-107,106.93159000(900),,92.90638(2)
41-Nb-108,107.93607600(900),,92.90638(2)
41-Nb-109,108.939140(460),,92.90638(2)
41-Nb-110,109.943840(900),,92.90638(2)
42-Mo-83,82.950250(430)#,,95.94(2)
42-Mo-84,83.941850(320)#,,95.94(2)
42-Mo-85,84.9382610(170),,95.94(2)
42-Mo-86,85.93117400(300),,95.94(2)
42-Mo-87,86.92819600(300),,95.94(2)
42-Mo-88,87.92196800(400),,95.94(2)
42-Mo-89,88.91946800(400),,95.94(2)
42-Mo-90,89.91393100(400),,95.94(2)
42-Mo-91,90.91174500(700),,95.94(2)
42-Mo-92,91.906807150(170),14.84(35),95.94(2)
42-Mo-93,92.906808770(190),,95.94(2)
42-Mo-94,93.905083590(150),9.25(12),95.94(2)
42-Mo-95,94.905837440(130),15.92(13),95.94(2)
42-Mo-96,95.904674770(130),16.68(2),95.94(2)
42-Mo-97,96.906016900(180),9.55(8),95.94(2)
42-Mo-98,97.905403610(190),24.13(31),95.94(2)
42-Mo-99,98.907707300(250),,95.94(2)
42-Mo-100,99.907468000(300),9.63(23),95.94(2)
42-Mo-101,100.910337600(300),,95.94(2)
42-Mo-102,101.91029400(900),,95.94(2)
42-Mo-103,102.9130920(100),,95.94(2)
42-Mo-104,103.9137470(100),,95.94(2)
42-Mo-105,104.9169820(100),,95.94(2)
42-Mo-106,105.9182730(100),,95.94(2)
42-Mo-107,106.9221200(100),,95.94(2)
42-Mo-108,107.9240480(100),,95.94(2)
42-Mo-109,108.9284380(120),,95.94(2)
42-Mo-110,109.9307180(260),,95.94(2)
42-Mo-111,110.9356520(140),,95.94(2)
42-Mo-112,111.938290(220)#,,95.94(2)
42-Mo-113,112.943480(320)#,,95.94(2)
43-Tc-85,84.950780(430)#,,[98]
43-Tc-86,85.944640(320)#,,[98]
43-Tc-87,86.93806700(500),,[98]
43-Tc-88,87.93379400(400),,[98]
43-Tc-89,88.92764900(400),,[98]
43-Tc-90,89.92407390(110),,[98]
43-Tc-91,90.91842500(250),,[98]
43-Tc-92,91.91527000(300),,[98]
43-Tc-93,92.91024510(110),,[98]
43-Tc-94,93.90965200(400),,[98]
43-Tc-95,94.90765200(500),,[98]
43-Tc-96,95.90786700(600),,[98]
43-Tc-97,96.90636100(400),,[98]
43-Tc-98,97.90721100(400),,[98]
43-Tc-99,98.90624970(100),,[98]
43-Tc-100,99.90765270(150),,[98]
43-Tc-101,100.9073050(260),,[98]
43-Tc-102,101.9092070(100),,[98]
43-Tc-103,102.9091740(110),,[98]
43-Tc-104,103.9114340(270),,[98]
43-Tc-105,104.9116600(400),,[98]
43-Tc-106,105.9143570(130),,[98]
43-Tc-107,106.91545800(900),,[98]
43-Tc-108,107.91849300(900),,[98]
43-Tc-109,108.9202540(100),,[98]
43-Tc-110,109.9237410(100),,[98]
43-Tc-111,110.9258990(110),,[98]
43-Tc-112,111.92994200(600),,[98]
43-Tc-113,112.93256900(400),,[98]
43-Tc-114,113.937090(470),,[98]
43-Tc-115,114.940100(210)#,,[98]
44-Ru-87,86.950910(430)#,,101.07(2)
44-Ru-88,87.941660(320)#,,101.07(2)
44-Ru-89,88.9373380(260),,101.07(2)
44-Ru-90,89.93034400(400),,101.07(2)
44-Ru-91,90.92674150(240),,101.07(2)
44-Ru-92,91.92023440(290),,101.07(2)
44-Ru-93,92.91710440(220),,101.07(2)
44-Ru-94,93.91134300(300),,101.07(2)
44-Ru-95,94.9104040(100),,101.07(2)
44-Ru-96,95.907588910(180),5.54(14),101.07(2)
44-Ru-97,96.90754580(300),,101.07(2)
44-Ru-98,97.90528700(700),1.87(3),101.07(2)
44-Ru-99,98.905930300(400),12.76(14),101.07(2)
44-Ru-100,99.904210500(400),12.60(7),101.07(2)
44-Ru-101,100.905573100(400),17.06(2),101.07(2)
44-Ru-102,101.904340300(400),31.55(14),101.07(2)
44-Ru-103,102.906314800(500),,101.07(2)
44-Ru-104,103.90542530(270),18.62(27),101.07(2)
44-Ru-105,104.90774550(270),,101.07(2)
44-Ru-106,105.90732800(600),,101.07(2)
44-Ru-107,106.90997000(900),,101.07(2)
44-Ru-108,107.91018600(900),,101.07(2)
44-Ru-109,108.9133240(100),,101.07(2)
44-Ru-110,109.9140390(100),,101.07(2)
44-Ru-111,110.9175680(100),,101.07(2)
44-Ru-112,111.9188070(100),,101.07(2)
44-Ru-113,112.9228500(400),,101.07(2)
44-Ru-114,113.92461400(400),,101.07(2)
44-Ru-115,114.9290330(270),,101.07(2)
44-Ru-116,115.93121900(400),,101.07(2)
44-Ru-117,116.936140(470),,101.07(2)
44-Ru-118,117.938810(220)#,,101.07(2)
45-Rh-89,88.950990(390)#,,102.90550(2)
45-Rh-90,89.944570(220)#,,102.90550(2)
45-Rh-91,90.937120(320)#,,102.90550(2)
45-Rh-92,91.93236800(500),,102.90550(2)
45-Rh-93,92.92591280(280),,102.90550(2)
45-Rh-94,93.92173000(400),,102.90550(2)
45-Rh-95,94.91589800(400),,102.90550(2)
45-Rh-96,95.9144520(110),,102.90550(2)
45-Rh-97,96.9113300(400),,102.90550(2)
45-Rh-98,97.9107080(130),,102.90550(2)
45-Rh-99,98.9081210(210),,102.90550(2)
45-Rh-100,99.9081140(190),,102.90550(2)
45-Rh-101,100.90615900(600),,102.90550(2)
45-Rh-102,101.90683400(700),,102.90550(2)
45-Rh-103,102.90549410(250),100,102.90550(2)
45-Rh-104,103.90664530(250),,102.90550(2)
45-Rh-105,104.90568780(270),,102.90550(2)
45-Rh-106,105.90728600(600),,102.90550(2)
45-Rh-107,106.9067480(130),,102.90550(2)
45-Rh-108,107.9087150(150),,102.90550(2)
45-Rh-109,108.90875000(400),,102.90550(2)
45-Rh-110,109.9110800(190),,102.90550(2)
45-Rh-111,110.91164300(700),,102.90550(2)
45-Rh-112,111.9144100(500),,102.90550(2)
45-Rh-113,112.91544000(800),,102.90550(2)
45-Rh-114,113.9187200(800),,102.90550(2)
45-Rh-115,114.92031200(800),,102.90550(2)
45-Rh-116,115.9240600(800),,102.90550(2)
45-Rh-117,116.9260360(100),,102.90550(2)
45-Rh-118,117.9303410(260),,102.90550(2)
45-Rh-119,118.9325570(100),,102.90550(2)
45-Rh-120,119.937070(220)#,,102.90550(2)
45-Rh-121,120.939610(670),,102.90550(2)
46-Pd-91,90.950440(450)#,,106.42(1)
46-Pd-92,91.941190(370),,106.42(1)
46-Pd-93,92.936680(400),,106.42(1)
46-Pd-94,93.92903600(500),,106.42(1)
46-Pd-95,94.92488900(300),,106.42(1)
46-Pd-96,95.91821400(500),,106.42(1)
46-Pd-97,96.91647200(500),,106.42(1)
46-Pd-98,97.91269800(500),,106.42(1)
46-Pd-99,98.91177300(500),,106.42(1)
46-Pd-100,99.9085200(190),,106.42(1)
46-Pd-101,100.90828500(500),,106.42(1)
46-Pd-102,101.905632300(400),1.02(1),106.42(1)
46-Pd-103,102.906111100(900),,106.42(1)
46-Pd-104,103.90403040(140),11.14(8),106.42(1)
46-Pd-105,104.90507950(120),22.33(8),106.42(1)
46-Pd-106,105.90348030(120),27.33(3),106.42(1)
46-Pd-107,106.90512810(130),,106.42(1)
46-Pd-108,107.90389180(120),26.46(9),106.42(1)
46-Pd-109,108.90595060(120),,106.42(1)
46-Pd-110,109.905172900(700),11.72(9),106.42(1)
46-Pd-111,110.907690400(800),,106.42(1)
46-Pd-112,111.90733100(700),,106.42(1)
46-Pd-113,112.91026200(700),,106.42(1)
46-Pd-114,113.91036900(700),,106.42(1)
46-Pd-115,114.9136590(150),,106.42(1)
46-Pd-116,115.91429800(800),,106.42(1)
46-Pd-117,116.91795600(800),,106.42(1)
46-Pd-118,117.91906730(270),,106.42(1)
46-Pd-119,118.92334100(900),,106.42(1)
46-Pd-120,119.92455170(250),,106.42(1)
46-Pd-121,120.92895000(400),,106.42(1)
46-Pd-122,121.9306320(210),,106.42(1)
46-Pd-123,122.935130(850),,106.42(1)
47-Ag-94,93.943740(430)#,,107.8682(2)
47-Ag-95,94.935690(430)#,,107.8682(2)
47-Ag-96,95.930740(100),,107.8682(2)
47-Ag-97,96.9238810(130),,107.8682(2)
47-Ag-98,97.9215600(400),,107.8682(2)
47-Ag-99,98.91764600(700),,107.8682(2)
47-Ag-100,99.91611500(500),,107.8682(2)
47-Ag-101,100.91268400(500),,107.8682(2)
47-Ag-102,101.91170500(900),,107.8682(2)
47-Ag-103,102.90896100(400),,107.8682(2)
47-Ag-104,103.90862400(500),,107.8682(2)
47-Ag-105,104.90652600(500),,107.8682(2)
47-Ag-106,105.90666300(300),,107.8682(2)
47-Ag-107,106.90509150(260),51.839(8),107.8682(2)
47-Ag-108,107.90595020(260),,107.8682(2)
47-Ag-109,108.90475580(140),48.161(8),107.8682(2)
47-Ag-110,109.90611070(140),,107.8682(2)
47-Ag-111,110.90529680(160),,107.8682(2)
47-Ag-112,111.90704850(260),,107.8682(2)
47-Ag-113,112.9065730(180),,107.8682(2)
47-Ag-114,113.90882300(500),,107.8682(2)
47-Ag-115,114.9087670(200),,107.8682(2)
47-Ag-116,115.91138700(400),,107.8682(2)
47-Ag-117,116.9117740(150),,107.8682(2)
47-Ag-118,117.91459550(270),,107.8682(2)
47-Ag-119,118.9155700(160),,107.8682(2)
47-Ag-120,119.91878500(500),,107.8682(2)
47-Ag-121,120.9201250(130),,107.8682(2)
47-Ag-122,121.9236600(400),,107.8682(2)
47-Ag-123,122.9253200(400),,107.8682(2)
47-Ag-124,123.928900(270),,107.8682(2)
47-Ag-125,124.930740(470),,107.8682(2)
47-Ag-126,125.934810(220)#,,107.8682(2)
47-Ag-127,126.937040(220)#,,107.8682(2)
48-Cd-96,95.940340(440)#,,112.411(8)
48-Cd-97,96.934800(450),,112.411(8)
48-Cd-98,97.9273900(600),,112.411(8)
48-Cd-99,98.92492580(170),,112.411(8)
48-Cd-100,99.92034880(180),,112.411(8)
48-Cd-101,100.91858620(160),,112.411(8)
48-Cd-102,101.91448180(180),,112.411(8)
48-Cd-103,102.91341690(190),,112.411(8)
48-Cd-104,103.90985620(180),,112.411(8)
48-Cd-105,104.90946390(150),,112.411(8)
48-Cd-106,105.90645980(120),1.25(6),112.411(8)
48-Cd-107,106.90661200(180),,112.411(8)
48-Cd-108,107.90418360(120),0.89(3),112.411(8)
48-Cd-109,108.90498670(160),,112.411(8)
48-Cd-110,109.903007500(400),12.49(18),112.411(8)
48-Cd-111,110.904183800(400),12.80(12),112.411(8)
48-Cd-112,111.902763900(270),24.13(21),112.411(8)
48-Cd-113,112.904408110(260),12.22(12),112.411(8)
48-Cd-114,113.903365000(300),28.73(42),112.411(8)
48-Cd-115,114.905437400(700),,112.411(8)
48-Cd-116,115.904763230(170),7.49(18),112.411(8)
48-Cd-117,116.90722600(110),,112.411(8)
48-Cd-118,117.9069220(210),,112.411(8)
48-Cd-119,118.9098500(400),,112.411(8)
48-Cd-120,119.90986800(400),,112.411(8)
48-Cd-121,120.91296370(210),,112.411(8)
48-Cd-122,121.91345910(250),,112.411(8)
48-Cd-123,122.91689250(290),,112.411(8)
48-Cd-124,123.91765980(280),,112.411(8)
48-Cd-125,124.92125800(300),,112.411(8)
48-Cd-126,125.92243030(250),,112.411(8)
48-Cd-127,126.92620300(700),,112.411(8)
48-Cd-128,127.92781700(700),,112.411(8)
48-Cd-129,128.93223600(600),,112.411(8)
48-Cd-130,129.9343880(240),,112.411(8)
49-In-98,97.942130(330)#,,114.818(3)
49-In-99,98.934110(320)#,,114.818(3)
49-In-100,99.93110190(240),,114.818(3)
49-In-101,100.9264140(130),,114.818(3)
49-In-102,101.92410600(500),,114.818(3)
49-In-103,102.9198790(100),,114.818(3)
49-In-104,103.91821500(600),,114.818(3)
49-In-105,104.9145020(110),,114.818(3)
49-In-106,105.9134640(130),,114.818(3)
49-In-107,106.9102870(100),,114.818(3)
49-In-108,107.90969400(900),,114.818(3)
49-In-109,108.90715000(400),,114.818(3)
49-In-110,109.9071710(120),,114.818(3)
49-In-111,110.90510700(400),,114.818(3)
49-In-112,111.90553900(500),,114.818(3)
49-In-113,112.904060450(200),4.29(5),114.818(3)
49-In-114,113.904916400(300),,114.818(3)
49-In-115,114.9038787730(120),95.71(5),114.818(3)
49-In-116,115.905259990(240),,114.818(3)
49-In-117,116.90451600(500),,114.818(3)
49-In-118,117.90635700(800),,114.818(3)
49-In-119,118.90585200(800),,114.818(3)
49-In-120,119.9079700(400),,114.818(3)
49-In-121,120.9078530(290),,114.818(3)
49-In-122,121.9102800(500),,114.818(3)
49-In-123,122.9104350(210),,114.818(3)
49-In-124,123.9131800(300),,114.818(3)
49-In-125,124.91367380(190),,114.818(3)
49-In-126,125.91646800(500),,114.818(3)
49-In-127,126.9174660(110),,114.818(3)
49-In-128,127.92035360(140),,114.818(3)
49-In-129,128.92180850(210),,114.818(3)
49-In-130,129.92495230(190),,114.818(3)
49-In-131,130.92697280(240),,114.818(3)
49-In-132,131.9330000(600),,114.818(3)
49-In-133,132.938070(220)#,,114.818(3)
49-In-134,133.944210(220)#,,114.818(3)
50-Sn-100,99.938650(260),,118.710(7)
50-Sn-101,100.935260(320),,118.710(7)
50-Sn-102,101.930290(110),,118.710(7)
50-Sn-103,102.927970(110)#,,118.710(7)
50-Sn-104,103.92310500(600),,118.710(7)
50-Sn-105,104.92126800(400),,118.710(7)
50-Sn-106,105.91695700(500),,118.710(7)
50-Sn-107,106.91571400(600),,118.710(7)
50-Sn-108,107.91189400(600),,118.710(7)
50-Sn-109,108.91129300(900),,118.710(7)
50-Sn-110,109.9078450(150),,118.710(7)
50-Sn-111,110.90774100(600),,118.710(7)
50-Sn-112,111.904824900(300),0.97(1),118.710(7)
50-Sn-113,112.90517590(170),,118.710(7)
50-Sn-114,113.9027801300(300),0.66(1),118.710(7)
50-Sn-115,114.9033446960(160),0.34(1),118.710(7)
50-Sn-116,115.901742830(100),14.54(9),118.710(7)
50-Sn-117,116.902954000(500),7.68(7),118.710(7)
50-Sn-118,117.901606600(500),24.22(9),118.710(7)
50-Sn-119,118.903311300(800),8.59(4),118.710(7)
50-Sn-120,119.90220260(100),32.58(9),118.710(7)
50-Sn-121,120.90424350(110),,118.710(7)
50-Sn-122,121.90344550(260),4.63(3),118.710(7)
50-Sn-123,122.90572710(270),,118.710(7)
50-Sn-124,123.90527960(140),5.79(5),118.710(7)
50-Sn-125,124.90778940(140),,118.710(7)
50-Sn-126,125.9076590(110),,118.710(7)
50-Sn-127,126.9103920(100),,118.710(7)
50-Sn-128,127.9105080(190),,118.710(7)
50-Sn-129,128.9134820(190),,118.710(7)
50-Sn-130,129.91397450(200),,118.710(7)
50-Sn-131,130.91705300(400),,118.710(7)
50-Sn-132,131.91782390(210),,118.710(7)
50-Sn-133,132.92391380(200),,118.710(7)
50-Sn-134,133.92868000(300),,118.710(7)
50-Sn-135,134.93490900(300),,118.710(7)
50-Sn-136,135.939700(220)#,,118.710(7)
50-Sn-137,136.946160(320)#,,118.710(7)
51-Sb-103,102.939160(320)#,,121.760(1)
51-Sb-104,103.936340(110)#,,121.760(1)
51-Sb-105,104.9312770(230),,121.760(1)
51-Sb-106,105.92863800(800),,121.760(1)
51-Sb-107,106.92415100(400),,121.760(1)
51-Sb-108,107.92222700(600),,121.760(1)
51-Sb-109,108.91814100(600),,121.760(1)
51-Sb-110,109.91685400(600),,121.760(1)
51-Sb-111,110.9132180(100),,121.760(1)
51-Sb-112,111.9124000(190),,121.760(1)
51-Sb-113,112.9093750(180),,121.760(1)
51-Sb-114,113.9092890(210),,121.760(1)
51-Sb-115,114.9065980(170),,121.760(1)
51-Sb-116,115.90679300(600),,121.760(1)
51-Sb-117,116.90484200(900),,121.760(1)
51-Sb-118,117.90553200(300),,121.760(1)
51-Sb-119,118.90394400(800),,121.760(1)
51-Sb-120,119.90508000(800),,121.760(1)
51-Sb-121,120.90381140(270),57.21(5),121.760(1)
51-Sb-122,121.90516930(270),,121.760(1)
51-Sb-123,122.90421530(150),42.79(5),121.760(1)
51-Sb-124,123.90593710(150),,121.760(1)
51-Sb-125,124.90525430(270),,121.760(1)
51-Sb-126,125.9072500(300),,121.760(1)
51-Sb-127,126.90692600(500),,121.760(1)
51-Sb-128,127.9091460(200),,121.760(1)
51-Sb-129,128.9091470(230),,121.760(1)
51-Sb-130,129.9116630(150),,121.760(1)
51-Sb-131,130.91198930(220),,121.760(1)
51-Sb-132,131.91450800(260),,121.760(1)
51-Sb-133,132.91527200(300),,121.760(1)
51-Sb-134,133.92053700(300),,121.760(1)
51-Sb-135,134.92518440(280),,121.760(1)
51-Sb-136,135.93074900(600),,121.760(1)
51-Sb-137,136.9355200(600),,121.760(1)
51-Sb-138,137.941330(320)#,,121.760(1)
51-Sb-139,138.946270(430)#,,121.760(1)
52-Te-106,105.937500(110),,127.60(3)
52-Te-107,106.934880(110)#,,127.60(3)
52-Te-108,107.92938000(600),,127.60(3)
52-Te-109,108.92730500(500),,127.60(3)
52-Te-110,109.92245800(700),,127.60(3)
52-Te-111,110.92100100(700),,127.60(3)
52-Te-112,111.91672800(900),,127.60(3)
52-Te-113,112.9158900(300),,127.60(3)
52-Te-114,113.9120880(260),,127.60(3)
52-Te-115,114.9119000(300),,127.60(3)
52-Te-116,115.9084660(260),,127.60(3)
52-Te-117,116.9086460(140),,127.60(3)
52-Te-118,117.9058600(200),,127.60(3)
52-Te-119,118.90640600(800),,127.60(3)
52-Te-120,119.90406580(190),0.09(1),127.60(3)
52-Te-121,120.9049450(280),,127.60(3)
52-Te-122,121.90304470(150),2.55(12),127.60(3)
52-Te-123,122.90427100(150),0.89(3),127.60(3)
52-Te-124,123.90281830(150),4.74(14),127.60(3)
52-Te-125,124.90443120(150),7.07(15),127.60(3)
52-Te-126,125.90331210(150),18.84(25),127.60(3)
52-Te-127,126.90522700(150),,127.60(3)
52-Te-128,127.904461200(800),31.74(8),127.60(3)
52-Te-129,128.906596400(800),,127.60(3)
52-Te-130,129.9062227450(120),34.08(62),127.60(3)
52-Te-131,130.9085222100(700),,127.60(3)
52-Te-132,131.90854700(400),,127.60(3)
52-Te-133,132.91096330(220),,127.60(3)
52-Te-134,133.91139640(290),,127.60(3)
52-Te-135,134.91655470(180),,127.60(3)
52-Te-136,135.92010120(240),,127.60(3)
52-Te-137,136.92559940(230),,127.60(3)
52-Te-138,137.92947200(400),,127.60(3)
52-Te-139,138.93536700(400),,127.60(3)
52-Te-140,139.9394870(150),,127.60(3)
52-Te-141,140.945600(430)#,,127.60(3)
52-Te-142,141.950030(540)#,,127.60(3)
53-I-108,107.943350(110)#,,126.90447(3)
53-I-109,108.93808600(700),,126.90447(3)
53-I-110,109.9350900(700),,126.90447(3)
53-I-111,110.93026900(500),,126.90447(3)
53-I-112,111.9280050(110),,126.90447(3)
53-I-113,112.92365000(900),,126.90447(3)
53-I-114,113.9220190(220),,126.90447(3)
53-I-115,114.9180500(300),,126.90447(3)
53-I-116,115.9168900(800),,126.90447(3)
53-I-117,116.9136460(270),,126.90447(3)
53-I-118,117.9130740(210),,126.90447(3)
53-I-119,118.9100610(230),,126.90447(3)
53-I-120,119.9100940(160),,126.90447(3)
53-I-121,120.90741100(500),,126.90447(3)
53-I-122,121.90759000(600),,126.90447(3)
53-I-123,122.90559000(400),,126.90447(3)
53-I-124,123.90621030(250),,126.90447(3)
53-I-125,124.90463060(150),,126.90447(3)
53-I-126,125.90562400(400),,126.90447(3)
53-I-127,126.90447300(400),100,126.90447(3)
53-I-128,127.90580900(400),,126.90447(3)
53-I-129,128.90498400(300),,126.90447(3)
53-I-130,129.90667000(300),,126.90447(3)
53-I-131,130.906126400(600),,126.90447(3)
53-I-132,131.90799400(400),,126.90447(3)
53-I-133,132.90782800(600),,126.90447(3)
53-I-134,133.90977600(500),,126.90447(3)
53-I-135,134.91005940(220),,126.90447(3)
53-I-136,135.9146050(150),,126.90447(3)
53-I-137,136.91802800(900),,126.90447(3)
53-I-138,137.92272600(600),,126.90447(3)
53-I-139,138.92649300(400),,126.90447(3)
53-I-140,139.9317160(130),,126.90447(3)
53-I-141,140.9356660(170),,126.90447(3)
53-I-142,141.94116700(500),,126.90447(3)
53-I-143,142.945480(220)#,,126.90447(3)
53-I-144,143.951340(430)#,,126.90447(3)
54-Xe-110,109.944260(110),,131.293(6)
54-Xe-111,110.941470(120)#,,131.293(6)
54-Xe-112,111.93555900(900),,131.293(6)
54-Xe-113,112.93322200(700),,131.293(6)
54-Xe-114,113.9279800(120),,131.293(6)
54-Xe-115,114.9262940(130),,131.293(6)
54-Xe-116,115.9215810(140),,131.293(6)
54-Xe-117,116.9203590(110),,131.293(6)
54-Xe-118,117.9161790(110),,131.293(6)
54-Xe-119,118.9154110(110),,131.293(6)
54-Xe-120,119.9117840(130),,131.293(6)
54-Xe-121,120.9114530(110),,131.293(6)
54-Xe-122,121.9083680(120),,131.293(6)
54-Xe-123,122.9084820(100),,131.293(6)
54-Xe-124,123.90588520(150),0.09(1),131.293(6)
54-Xe-125,124.90638760(150),,131.293(6)
54-Xe-126,125.90429742200(600),0.09(1),131.293(6)
54-Xe-127,126.90518400(400),,131.293(6)
54-Xe-128,127.90353075300(600),1.92(3),131.293(6)
54-Xe-129,128.90478085700(500),26.44(24),131.293(6)
54-Xe-130,129.9035093470(100),4.08(2),131.293(6)
54-Xe-131,130.90508412800(500),21.18(3),131.293(6)
54-Xe-132,131.90415508300(500),26.89(6),131.293(6)
54-Xe-133,132.90591070(260),,131.293(6)
54-Xe-134,133.90539303000(600),10.44(10),131.293(6)
54-Xe-135,134.90723100(400),,131.293(6)
54-Xe-136,135.90721447400(700),8.87(16),131.293(6)
54-Xe-137,136.911557770(110),,131.293(6)
54-Xe-138,137.91414600(300),,131.293(6)
54-Xe-139,138.91879220(230),,131.293(6)
54-Xe-140,139.92164580(250),,131.293(6)
54-Xe-141,140.92678700(300),,131.293(6)
54-Xe-142,141.92997310(290),,131.293(6)
54-Xe-143,142.93537000(500),,131.293(6)
54-Xe-144,143.93894500(600),,131.293(6)
54-Xe-145,144.9447200(120),,131.293(6)
54-Xe-146,145.9485180(260),,131.293(6)
54-Xe-147,146.954480(220)#,,131.293(6)
55-Cs-112,111.950170(120)#,,132.90545(2)
55-Cs-113,112.94442800(900),,132.90545(2)
55-Cs-114,113.9412900(900),,132.90545(2)
55-Cs-115,114.935910(110)#,,132.90545(2)
55-Cs-116,115.933400(110)#,,132.90545(2)
55-Cs-117,116.9286200(700),,132.90545(2)
55-Cs-118,117.9265600(140),,132.90545(2)
55-Cs-119,118.9223770(150),,132.90545(2)
55-Cs-120,119.9206770(110),,132.90545(2)
55-Cs-121,120.9172270(150),,132.90545(2)
55-Cs-122,121.9161100(400),,132.90545(2)
55-Cs-123,122.9129960(130),,132.90545(2)
55-Cs-124,123.9122470(100),,132.90545(2)
55-Cs-125,124.90972600(800),,132.90545(2)
55-Cs-126,125.9094460(110),,132.90545(2)
55-Cs-127,126.90741800(600),,132.90545(2)
55-Cs-128,127.90774800(600),,132.90545(2)
55-Cs-129,128.90606600(500),,132.90545(2)
55-Cs-130,129.90670900(900),,132.90545(2)
55-Cs-131,130.905468460(190),,132.90545(2)
55-Cs-132,131.90643770(110),,132.90545(2)
55-Cs-133,132.90545195900(900),100,132.90545(2)
55-Cs-134,133.9067185020(180),,132.90545(2)
55-Cs-135,134.905976900(400),,132.90545(2)
55-Cs-136,135.90731140(200),,132.90545(2)
55-Cs-137,136.907089300(300),,132.90545(2)
55-Cs-138,137.9110170(100),,132.90545(2)
55-Cs-139,138.91336400(300),,132.90545(2)
55-Cs-140,139.91728400(900),,132.90545(2)
55-Cs-141,140.9200450(100),,132.90545(2)
55-Cs-142,141.92430000(800),,132.90545(2)
55-Cs-143,142.92734700(800),,132.90545(2)
55-Cs-144,143.9320750(220),,132.90545(2)
55-Cs-145,144.9355290(100),,132.90545(2)
55-Cs-146,145.94062200(300),,132.90545(2)
55-Cs-147,146.94426200(900),,132.90545(2)
55-Cs-148,147.9496390(140),,132.90545(2)
55-Cs-149,148.953520(430)#,,132.90545(2)
55-Cs-150,149.959020(430)#,,132.90545(2)
55-Cs-151,150.963200(540)#,,132.90545(2)
56-Ba-114,113.950720(110),,137.327(7)
56-Ba-115,114.947480(220)#,,137.327(7)
56-Ba-116,115.941620(220)#,,137.327(7)
56-Ba-117,116.938320(270),,137.327(7)
56-Ba-118,117.933230(220)#,,137.327(7)
56-Ba-119,118.930660(210),,137.327(7)
56-Ba-120,119.926040(320),,137.327(7)
56-Ba-121,120.924050(150),,137.327(7)
56-Ba-122,121.9199000(300),,137.327(7)
56-Ba-123,122.9187810(130),,137.327(7)
56-Ba-124,123.9150940(130),,137.327(7)
56-Ba-125,124.9144720(120),,137.327(7)
56-Ba-126,125.9112500(130),,137.327(7)
56-Ba-127,126.9110910(120),,137.327(7)
56-Ba-128,127.90835240(170),,137.327(7)
56-Ba-129,128.9086830(110),,137.327(7)
56-Ba-130,129.906326000(300),0.106(1),137.327(7)
56-Ba-131,130.906946300(400),,137.327(7)
56-Ba-132,131.90506120(110),0.101(1),137.327(7)
56-Ba-133,132.90600740(110),,137.327(7)
56-Ba-134,133.904508250(270),2.417(18),137.327(7)
56-Ba-135,134.905688450(260),6.592(12),137.327(7)
56-Ba-136,135.904575800(260),7.854(24),137.327(7)
56-Ba-137,136.905827210(270),11.232(24),137.327(7)
56-Ba-138,137.905247060(270),71.698(42),137.327(7)
56-Ba-139,138.908841160(270),,137.327(7)
56-Ba-140,139.91060800(800),,137.327(7)
56-Ba-141,140.91440400(600),,137.327(7)
56-Ba-142,141.91643300(600),,137.327(7)
56-Ba-143,142.92062500(700),,137.327(7)
56-Ba-144,143.92295500(800),,137.327(7)
56-Ba-145,144.92751800(900),,137.327(7)
56-Ba-146,145.93036320(190),,137.327(7)
56-Ba-147,146.9353040(210),,137.327(7)
56-Ba-148,147.93822300(160),,137.327(7)
56-Ba-149,148.94328400(270),,137.327(7)
56-Ba-150,149.94644100(600),,137.327(7)
56-Ba-151,150.951760(430)#,,137.327(7)
56-Ba-152,151.955330(430)#,,137.327(7)
56-Ba-153,152.960850(430)#,,137.327(7)
57-La-117,116.950330(220)#,,138.9055(2)
57-La-118,117.946730(320)#,,138.9055(2)
57-La-119,118.940930(320)#,,138.9055(2)
57-La-120,119.938200(320)#,,138.9055(2)
57-La-121,120.933240(320)#,,138.9055(2)
57-La-122,121.930710(320)#,,138.9055(2)
57-La-123,122.926300(210)#,,138.9055(2)
57-La-124,123.9245700(600),,138.9055(2)
57-La-125,124.9208160(280),,138.9055(2)
57-La-126,125.919510(100),,138.9055(2)
57-La-127,126.9163750(280),,138.9055(2)
57-La-128,127.9155900(600),,138.9055(2)
57-La-129,128.9126960(230),,138.9055(2)
57-La-130,129.9123690(280),,138.9055(2)
57-La-131,130.9100700(300),,138.9055(2)
57-La-132,131.9101200(400),,138.9055(2)
57-La-133,132.9082200(300),,138.9055(2)
57-La-134,133.9085140(210),,138.9055(2)
57-La-135,134.9069840(100),,138.9055(2)
57-La-136,135.9076300(600),,138.9055(2)
57-La-137,136.90645040(180),,138.9055(2)
57-La-138,137.907124000(400),0.090(1),138.9055(2)
57-La-139,138.906362900(700),99.910(1),138.9055(2)
57-La-140,139.909487300(700),,138.9055(2)
57-La-141,140.91097100(400),,138.9055(2)
57-La-142,141.91409100(700),,138.9055(2)
57-La-143,142.91607900(800),,138.9055(2)
57-La-144,143.9196460(140),,138.9055(2)
57-La-145,144.9218080(130),,138.9055(2)
57-La-146,145.92568800(180),,138.9055(2)
57-La-147,146.9284180(120),,138.9055(2)
57-La-148,147.9326790(210),,138.9055(2)
57-La-149,148.935350(210),,138.9055(2)
57-La-150,149.93954750(270),,138.9055(2)
57-La-151,150.942770(470),,138.9055(2)
57-La-152,151.947090(320)#,,138.9055(2)
57-La-153,152.950550(320)#,,138.9055(2)
57-La-154,153.955420(320)#,,138.9055(2)
57-La-155,154.959280(430)#,,138.9055(2)
58-Ce-119,118.952960(540)#,,140.116(1)
58-Ce-120,119.946610(540)#,,140.116(1)
58-Ce-121,120.943440(430)#,,140.116(1)
58-Ce-122,121.937870(430)#,,140.116(1)
58-Ce-123,122.935280(320)#,,140.116(1)
58-Ce-124,123.930310(320)#,,140.116(1)
58-Ce-125,124.928440(210)#,,140.116(1)
58-Ce-126,125.9239700(300),,140.116(1)
58-Ce-127,126.9227300(300),,140.116(1)
58-Ce-128,127.9189100(300),,140.116(1)
58-Ce-129,128.9181000(300),,140.116(1)
58-Ce-130,129.9147400(300),,140.116(1)
58-Ce-131,130.9144300(400),,140.116(1)
58-Ce-132,131.9114660(220),,140.116(1)
58-Ce-133,132.9115200(180),,140.116(1)
58-Ce-134,133.9089280(220),,140.116(1)
58-Ce-135,134.9091610(110),,140.116(1)
58-Ce-136,135.907129300(300),0.185(2),140.116(1)
58-Ce-137,136.907762400(400),,140.116(1)
58-Ce-138,137.905994200(500),0.251(2),140.116(1)
58-Ce-139,138.90664700(220),,140.116(1)
58-Ce-140,139.90544840(140),88.450(51),140.116(1)
58-Ce-141,140.90828600(140),,140.116(1)
58-Ce-142,141.90925020(260),11.114(51),140.116(1)
58-Ce-143,142.91239200(260),,140.116(1)
58-Ce-144,143.91365300(300),,140.116(1)
58-Ce-145,144.9172700(400),,140.116(1)
58-Ce-146,145.9188120(160),,140.116(1)
58-Ce-147,146.92269000(900),,140.116(1)
58-Ce-148,147.9244240(120),,140.116(1)
58-Ce-149,148.9284270(110),,140.116(1)
58-Ce-150,149.9303840(130),,140.116(1)
58-Ce-151,150.9342720(190),,140.116(1)
58-Ce-152,151.936680(220)#,,140.116(1)
58-Ce-153,152.941050(220)#,,140.116(1)
58-Ce-154,153.943940(220)#,,140.116(1)
58-Ce-155,154.948710(320)#,,140.116(1)
58-Ce-156,155.951880(320)#,,140.116(1)
58-Ce-157,156.957130(430)#,,140.116(1)
59-Pr-121,120.955390(540)#,,140.90765(2)
59-Pr-122,121.951930(540)#,,140.90765(2)
59-Pr-123,122.946080(430)#,,140.90765(2)
59-Pr-124,123.942940(430)#,,140.90765(2)
59-Pr-125,124.937660(320)#,,140.90765(2)
59-Pr-126,125.935240(210)#,,140.90765(2)
59-Pr-127,126.930710(210)#,,140.90765(2)
59-Pr-128,127.9287900(300),,140.90765(2)
59-Pr-129,128.9251000(300),,140.90765(2)
59-Pr-130,129.9235900(700),,140.90765(2)
59-Pr-131,130.9202300(500),,140.90765(2)
59-Pr-132,131.9192400(300),,140.90765(2)
59-Pr-133,132.9163310(130),,140.90765(2)
59-Pr-134,133.9156970(220),,140.90765(2)
59-Pr-135,134.9131120(130),,140.90765(2)
59-Pr-136,135.9126770(120),,140.90765(2)
59-Pr-137,136.91067900(900),,140.90765(2)
59-Pr-138,137.9107570(110),,140.90765(2)
59-Pr-139,138.90893300(400),,140.90765(2)
59-Pr-140,139.90908600(700),,140.90765(2)
59-Pr-141,140.90765960(160),100,140.90765(2)
59-Pr-142,141.91005160(160),,140.90765(2)
59-Pr-143,142.91082260(190),,140.90765(2)
59-Pr-144,143.91331070(290),,140.90765(2)
59-Pr-145,144.91451800(800),,140.90765(2)
59-Pr-146,145.9176900(400),,140.90765(2)
59-Pr-147,146.9190070(170),,140.90765(2)
59-Pr-148,147.9221300(160),,140.90765(2)
59-Pr-149,148.9237360(110),,140.90765(2)
59-Pr-150,149.9266760(100),,140.90765(2)
59-Pr-151,150.9283090(130),,140.90765(2)
59-Pr-152,151.9315530(200),,140.90765(2)
59-Pr-153,152.9339040(130),,140.90765(2)
59-Pr-154,153.937890(110),,140.90765(2)
59-Pr-155,154.9405090(180),,140.90765(2)
59-Pr-156,155.94476690(110),,140.90765(2)
59-Pr-157,156.94800300(300),,140.90765(2)
59-Pr-158,157.952600(320)#,,140.90765(2)
59-Pr-159,158.956230(430)#,,140.90765(2)
60-Nd-126,125.942690(320)#,,144.24(3)
60-Nd-127,126.939980(320)#,,144.24(3)
60-Nd-128,127.935020(220)#,,144.24(3)
60-Nd-129,128.933040(220)#,,144.24(3)
60-Nd-130,129.9285100(300),,144.24(3)
60-Nd-131,130.9272480(300),,144.24(3)
60-Nd-132,131.9233210(260),,144.24(3)
60-Nd-133,132.9223500(500),,144.24(3)
60-Nd-134,133.9187900(130),,144.24(3)
60-Nd-135,134.9181810(210),,144.24(3)
60-Nd-136,135.9149760(130),,144.24(3)
60-Nd-137,136.9145630(130),,144.24(3)
60-Nd-138,137.9119510(120),,144.24(3)
60-Nd-139,138.9119510(300),,144.24(3)
60-Nd-140,139.90954600(400),,144.24(3)
60-Nd-141,140.90961700(300),,144.24(3)
60-Nd-142,141.90772880(130),27.2(5),144.24(3)
60-Nd-143,142.90981980(130),12.2(2),144.24(3)
60-Nd-144,143.91009280(130),23.8(3),144.24(3)
60-Nd-145,144.91257920(140),8.3(1),144.24(3)
60-Nd-146,145.91312250(140),17.2(3),144.24(3)
60-Nd-147,146.91610600(140),,144.24(3)
60-Nd-148,147.91689900(220),5.7(1),144.24(3)
60-Nd-149,148.92015460(220),,144.24(3)
60-Nd-150,149.92090130(120),5.6(2),144.24(3)
60-Nd-151,150.92383940(120),,144.24(3)
60-Nd-152,151.9246910(260),,144.24(3)
60-Nd-153,152.92771790(290),,144.24(3)
60-Nd-154,153.92959740(110),,144.24(3)
60-Nd-155,154.9331360(100),,144.24(3)
60-Nd-156,155.93537040(140),,144.24(3)
60-Nd-157,156.93935110(230),,144.24(3)
60-Nd-158,157.94220560(140),,144.24(3)
60-Nd-159,158.9466200(300),,144.24(3)
60-Nd-160,159.9498400(500),,144.24(3)
60-Nd-161,160.954660(430)#,,144.24(3)
61-Pm-128,127.948230(320)#,,[145]
61-Pm-129,128.942910(320)#,,[145]
61-Pm-130,129.940450(220)#,,[145]
61-Pm-131,130.935830(220)#,,[145]
61-Pm-132,131.933840(160)#,,[145]
61-Pm-133,132.9297800(500),,[145]
61-Pm-134,133.9283300(500),,[145]
61-Pm-135,134.9247900(900),,[145]
61-Pm-136,135.9236000(700),,[145]
61-Pm-137,136.9204800(140),,[145]
61-Pm-138,137.9195760(120),,[145]
61-Pm-139,138.9167990(150),,[145]
61-Pm-140,139.9160360(260),,[145]
61-Pm-141,140.9135550(150),,[145]
61-Pm-142,141.9128910(250),,[145]
61-Pm-143,142.91093800(300),,[145]
61-Pm-144,143.91259600(300),,[145]
61-Pm-145,144.91275600(300),,[145]
61-Pm-146,145.91470200(500),,[145]
61-Pm-147,146.91514490(140),,[145]
61-Pm-148,147.91748100(600),,[145]
61-Pm-149,148.91834150(230),,[145]
61-Pm-150,149.9209900(220),,[145]
61-Pm-151,150.92121700(500),,[145]
61-Pm-152,151.9235050(280),,[145]
61-Pm-153,152.9241560(100),,[145]
61-Pm-154,153.9267130(270),,[145]
61-Pm-155,154.92813700(500),,[145]
61-Pm-156,155.93111410(130),,[145]
61-Pm-157,156.93312100(800),,[145]
61-Pm-158,157.93654690(100),,[145]
61-Pm-159,158.9392860(110),,[145]
61-Pm-160,159.94321530(220),,[145]
61-Pm-161,160.9462300(100),,[145]
61-Pm-162,161.950570(320)#,,[145]
61-Pm-163,162.953880(430)#,,[145]
62-Sm-130,129.948790(430)#,,150.36(3)
62-Sm-131,130.946020(430)#,,150.36(3)
62-Sm-132,131.940810(320)#,,150.36(3)
62-Sm-133,132.938560(320)#,,150.36(3)
62-Sm-134,133.934110(210)#,,150.36(3)
62-Sm-135,134.932520(170),,150.36(3)
62-Sm-136,135.9282760(130),,150.36(3)
62-Sm-137,136.9270100(300),,150.36(3)
62-Sm-138,137.9232440(130),,150.36(3)
62-Sm-139,138.9222970(120),,150.36(3)
62-Sm-140,139.9189950(130),,150.36(3)
62-Sm-141,140.91848200(900),,150.36(3)
62-Sm-142,141.91520940(200),,150.36(3)
62-Sm-143,142.91463480(300),,150.36(3)
62-Sm-144,143.91200630(160),3.07(7),150.36(3)
62-Sm-145,144.91341720(160),,150.36(3)
62-Sm-146,145.91304700(300),,150.36(3)
62-Sm-147,146.91490440(140),14.99(18),150.36(3)
62-Sm-148,147.91482920(130),11.24(10),150.36(3)
62-Sm-149,148.91719120(120),13.82(7),150.36(3)
62-Sm-150,149.91728200(120),7.38(1),150.36(3)
62-Sm-151,150.91993890(120),,150.36(3)
62-Sm-152,151.91973860(110),26.75(16),150.36(3)
62-Sm-153,152.92210360(110),,150.36(3)
62-Sm-154,153.92221580(140),22.75(29),150.36(3)
62-Sm-155,154.92464660(140),,150.36(3)
62-Sm-156,155.92553800(900),,150.36(3)
62-Sm-157,156.92841900(500),,150.36(3)
62-Sm-158,157.92994900(500),,150.36(3)
62-Sm-159,158.93321700(600),,150.36(3)
62-Sm-160,159.93533700(210),,150.36(3)
62-Sm-161,160.93916000(700),,150.36(3)
62-Sm-162,161.94162200(400),,150.36(3)
62-Sm-163,162.94567900(800),,150.36(3)
62-Sm-164,163.94855000(400),,150.36(3)
62-Sm-165,164.953290(430)#,,150.36(3)
63-Eu-132,131.954700(430)#,,151.964(1)
63-Eu-133,132.949290(320)#,,151.964(1)
63-Eu-134,133.946540(320)#,,151.964(1)
63-Eu-135,134.941870(210)#,,151.964(1)
63-Eu-136,135.939620(210)#,,151.964(1)
63-Eu-137,136.93543100(500),,151.964(1)
63-Eu-138,137.9337100(300),,151.964(1)
63-Eu-139,138.9297920(140),,151.964(1)
63-Eu-140,139.9280900(600),,151.964(1)
63-Eu-141,140.9249320(140),,151.964(1)
63-Eu-142,141.9234500(300),,151.964(1)
63-Eu-143,142.9202990(120),,151.964(1)
63-Eu-144,143.9188190(120),,151.964(1)
63-Eu-145,144.91627300(300),,151.964(1)
63-Eu-146,145.91721100(600),,151.964(1)
63-Eu-147,146.91675240(280),,151.964(1)
63-Eu-148,147.9180910(110),,151.964(1)
63-Eu-149,148.91793700(400),,151.964(1)
63-Eu-150,149.91970700(700),,151.964(1)
63-Eu-151,150.91985660(130),47.81(3),151.964(1)
63-Eu-152,151.92175100(130),,151.964(1)
63-Eu-153,152.92123680(130),52.19(3),151.964(1)
63-Eu-154,153.92298570(130),,151.964(1)
63-Eu-155,154.92289980(130),,151.964(1)
63-Eu-156,155.92476300(400),,151.964(1)
63-Eu-157,156.92543300(500),,151.964(1)
63-Eu-158,157.92778220(220),,151.964(1)
63-Eu-159,158.92910000(500),,151.964(1)
63-Eu-160,159.93183700(100),,151.964(1)
63-Eu-161,160.9336640(110),,151.964(1)
63-Eu-162,161.93695830(140),,151.964(1)
63-Eu-163,162.93926550(100),,151.964(1)
63-Eu-164,163.94285290(220),,151.964(1)
63-Eu-165,164.94554000(600),,151.964(1)
63-Eu-166,165.949810(110)#,,151.964(1)
63-Eu-167,166.953010(430)#,,151.964(1)
64-Gd-136,135.947300(320)#,,157.25(3)
64-Gd-137,136.945020(320)#,,157.25(3)
64-Gd-138,137.940250(220)#,,157.25(3)
64-Gd-139,138.938130(210)#,,157.25(3)
64-Gd-140,139.9336700(300),,157.25(3)
64-Gd-141,140.9321260(210),,157.25(3)
64-Gd-142,141.9281200(300),,157.25(3)
64-Gd-143,142.926750(220),,157.25(3)
64-Gd-144,143.9229600(300),,157.25(3)
64-Gd-145,144.9217100(210),,157.25(3)
64-Gd-146,145.91831900(400),,157.25(3)
64-Gd-147,146.91910100(200),,157.25(3)
64-Gd-148,147.91812140(160),,157.25(3)
64-Gd-149,148.91934800(400),,157.25(3)
64-Gd-150,149.91866400(700),,157.25(3)
64-Gd-151,150.92035500(300),,157.25(3)
64-Gd-152,151.91979840(110),0.20(1),157.25(3)
64-Gd-153,152.92175690(110),,157.25(3)
64-Gd-154,153.92087300(110),2.18(3),157.25(3)
64-Gd-155,154.92262940(110),14.80(12),157.25(3)
64-Gd-156,155.92213010(110),20.47(9),157.25(3)
64-Gd-157,156.92396740(100),15.65(2),157.25(3)
64-Gd-158,157.92411120(100),24.84(7),157.25(3)
64-Gd-159,158.92639580(110),,157.25(3)
64-Gd-160,159.92706120(120),21.86(19),157.25(3)
64-Gd-161,160.92967630(160),,157.25(3)
64-Gd-162,161.93099200(400),,157.25(3)
64-Gd-163,162.934096600(900),,157.25(3)
64-Gd-164,163.93591620(110),,157.25(3)
64-Gd-165,164.93931710(140),,157.25(3)
64-Gd-166,165.94163040(170),,157.25(3)
64-Gd-167,166.94549000(600),,157.25(3)
64-Gd-168,167.948310(320)#,,157.25(3)
64-Gd-169,168.952880(430)#,,157.25(3)
65-Tb-138,137.953190(320)#,,158.92534(2)
65-Tb-139,138.948330(320)#,,158.92534(2)
65-Tb-140,139.945810(860),,158.92534(2)
65-Tb-141,140.941450(110),,158.92534(2)
65-Tb-142,141.939280(750),,158.92534(2)
65-Tb-143,142.9351400(600),,158.92534(2)
65-Tb-144,143.9330500(300),,158.92534(2)
65-Tb-145,144.928720(120),,158.92534(2)
65-Tb-146,145.9272500(500),,158.92534(2)
65-Tb-147,146.92405500(900),,158.92534(2)
65-Tb-148,147.9242750(130),,158.92534(2)
65-Tb-149,148.92325400(400),,158.92534(2)
65-Tb-150,149.92366500(800),,158.92534(2)
65-Tb-151,150.92310900(400),,158.92534(2)
65-Tb-152,151.9240800(400),,158.92534(2)
65-Tb-153,152.92344200(400),,158.92534(2)
65-Tb-154,153.9246800(500),,158.92534(2)
65-Tb-155,154.9235100(110),,158.92534(2)
65-Tb-156,155.92475400(400),,158.92534(2)
65-Tb-157,156.92403190(110),,158.92534(2)
65-Tb-158,157.92541990(140),,158.92534(2)
65-Tb-159,158.92535370(120),100,158.92534(2)
65-Tb-160,159.92717460(120),,158.92534(2)
65-Tb-161,160.92757680(130),,158.92534(2)
65-Tb-162,161.92927540(220),,158.92534(2)
65-Tb-163,162.93065400(400),,158.92534(2)
65-Tb-164,163.93332760(200),,158.92534(2)
65-Tb-165,164.93495520(170),,158.92534(2)
65-Tb-166,165.93793970(160),,158.92534(2)
65-Tb-167,166.94000700(210),,158.92534(2)
65-Tb-168,167.94333700(500),,158.92534(2)
65-Tb-169,168.945810(320)#,,158.92534(2)
65-Tb-170,169.949860(320)#,,158.92534(2)
65-Tb-171,170.953010(430)#,,158.92534(2)
66-Dy-140,139.954020(430)#,,162.500(1)
66-Dy-141,140.951280(320)#,,162.500(1)
66-Dy-142,141.946190(780)#,,162.500(1)
66-Dy-143,142.9439940(140),,162.500(1)
66-Dy-144,143.93927000(800),,162.500(1)
66-Dy-145,144.93747400(700),,162.500(1)
66-Dy-146,145.93284500(700),,162.500(1)
66-Dy-147,146.9310830(100),,162.500(1)
66-Dy-148,147.92715000(900),,162.500(1)
66-Dy-149,148.9273280(100),,162.500(1)
66-Dy-150,149.92559300(500),,162.500(1)
66-Dy-151,150.92619100(300),,162.500(1)
66-Dy-152,151.92472500(500),,162.500(1)
66-Dy-153,152.92577200(400),,162.500(1)
66-Dy-154,153.92442900(800),,162.500(1)
66-Dy-155,154.9257580(100),,162.500(1)
66-Dy-156,155.92428360(110),0.06(1),162.500(1)
66-Dy-157,156.92547000(600),,162.500(1)
66-Dy-158,157.92441480(250),0.10(1),162.500(1)
66-Dy-159,158.92574590(150),,162.500(1)
66-Dy-160,159.925203600(800),2.34(8),162.500(1)
66-Dy-161,160.926939400(700),18.91(24),162.500(1)
66-Dy-162,161.926804500(700),25.51(26),162.500(1)
66-Dy-163,162.928737200(700),24.90(16),162.500(1)
66-Dy-164,163.929180800(700),28.18(37),162.500(1)
66-Dy-165,164.931709400(700),,162.500(1)
66-Dy-166,165.932812800(900),,162.500(1)
66-Dy-167,166.93568200(400),,162.500(1)
66-Dy-168,167.937130(150),,162.500(1)
66-Dy-169,168.940320(320),,162.500(1)
66-Dy-170,169.942340(220)#,,162.500(1)
66-Dy-171,170.946310(220)#,,162.500(1)
66-Dy-172,171.948730(320)#,,162.500(1)
66-Dy-173,172.953040(430)#,,162.500(1)
67-Ho-142,141.960010(430)#,,164.93032(2)
67-Ho-143,142.954860(320)#,,164.93032(2)
67-Ho-144,143.95211000(900),,164.93032(2)
67-Ho-145,144.94726700(800),,164.93032(2)
67-Ho-146,145.94499400(700),,164.93032(2)
67-Ho-147,146.94014200(500),,164.93032(2)
67-Ho-148,147.9377400(900),,164.93032(2)
67-Ho-149,148.9338200(130),,164.93032(2)
67-Ho-150,149.9334980(150),,164.93032(2)
67-Ho-151,150.93169800(900),,164.93032(2)
67-Ho-152,151.9317180(130),,164.93032(2)
67-Ho-153,152.93020700(500),,164.93032(2)
67-Ho-154,153.93060700(900),,164.93032(2)
67-Ho-155,154.9291030(190),,164.93032(2)
67-Ho-156,155.9296400(400),,164.93032(2)
67-Ho-157,156.9282520(250),,164.93032(2)
67-Ho-158,157.9289450(290),,164.93032(2)
67-Ho-159,158.92771900(300),,164.93032(2)
67-Ho-160,159.9287360(160),,164.93032(2)
67-Ho-161,160.92786180(230),,164.93032(2)
67-Ho-162,161.92910300(300),,164.93032(2)
67-Ho-163,162.928740300(700),,164.93032(2)
67-Ho-164,163.93024050(150),,164.93032(2)
67-Ho-165,164.930329100(800),100,164.93032(2)
67-Ho-166,165.932291200(800),,164.93032(2)
67-Ho-167,166.93314000(600),,164.93032(2)
67-Ho-168,167.9355200(300),,164.93032(2)
67-Ho-169,168.9368800(220),,164.93032(2)
67-Ho-170,169.9396300(500),,164.93032(2)
67-Ho-171,170.941470(640),,164.93032(2)
67-Ho-172,171.944730(210)#,,164.93032(2)
67-Ho-173,172.947020(320)#,,164.93032(2)
67-Ho-174,173.950760(320)#,,164.93032(2)
67-Ho-175,174.953520(430)#,,164.93032(2)
68-Er-144,143.960700(210)#,,167.259(3)
68-Er-145,144.957870(220)#,,167.259(3)
68-Er-146,145.95241800(700),,167.259(3)
68-Er-147,146.9499600(400),,167.259(3)
68-Er-148,147.9447350(110),,167.259(3)
68-Er-149,148.9423100(300),,167.259(3)
68-Er-150,149.9379160(180),,167.259(3)
68-Er-151,150.9374490(180),,167.259(3)
68-Er-152,151.93505000(900),,167.259(3)
68-Er-153,152.9350860(100),,167.259(3)
68-Er-154,153.93279100(500),,167.259(3)
68-Er-155,154.93321600(700),,167.259(3)
68-Er-156,155.9310660(260),,167.259(3)
68-Er-157,156.9319230(280),,167.259(3)
68-Er-158,157.9298930(270),,167.259(3)
68-Er-159,158.93069100(400),,167.259(3)
68-Er-160,159.9290770(260),,167.259(3)
68-Er-161,160.93000400(900),,167.259(3)
68-Er-162,161.928787300(800),0.14(1),167.259(3)
68-Er-163,162.93004000(500),,167.259(3)
68-Er-164,163.929207700(800),1.61(3),167.259(3)
68-Er-165,164.93073350(100),,167.259(3)
68-Er-166,165.930301100(400),33.61(35),167.259(3)
68-Er-167,166.932056200(300),22.93(17),167.259(3)
68-Er-168,167.932378280(280),26.78(26),167.259(3)
68-Er-169,168.934598400(300),,167.259(3)
68-Er-170,169.93547190(150),14.93(27),167.259(3)
68-Er-171,170.93803740(150),,167.259(3)
68-Er-172,171.93936300(400),,167.259(3)
68-Er-173,172.942400(210)#,,167.259(3)
68-Er-174,173.944230(320)#,,167.259(3)
68-Er-175,174.947770(430)#,,167.259(3)
68-Er-176,175.949940(430)#,,167.259(3)
68-Er-177,176.953990(540)#,,167.259(3)
69-Tm-146,145.966660(220)#,,168.93421(2)
69-Tm-147,146.96138000(700),,168.93421(2)
69-Tm-148,147.9583840(110),,168.93421(2)
69-Tm-149,148.952830(220)#,,168.93421(2)
69-Tm-150,149.950090(210)#,,168.93421(2)
69-Tm-151,150.9454940(210),,168.93421(2)
69-Tm-152,151.9444800(600),,168.93421(2)
69-Tm-153,152.9420580(130),,168.93421(2)
69-Tm-154,153.9415700(150),,168.93421(2)
69-Tm-155,154.9392100(110),,168.93421(2)
69-Tm-156,155.9389860(150),,168.93421(2)
69-Tm-157,156.9369700(300),,168.93421(2)
69-Tm-158,157.9369800(270),,168.93421(2)
69-Tm-159,158.9349800(300),,168.93421(2)
69-Tm-160,159.9352600(400),,168.93421(2)
69-Tm-161,160.9335500(300),,168.93421(2)
69-Tm-162,161.9340010(280),,168.93421(2)
69-Tm-163,162.93265800(600),,168.93421(2)
69-Tm-164,163.9335380(270),,168.93421(2)
69-Tm-165,164.93244180(180),,168.93421(2)
69-Tm-166,165.9335620(120),,168.93421(2)
69-Tm-167,166.93285720(140),,168.93421(2)
69-Tm-168,167.93417850(180),,168.93421(2)
69-Tm-169,168.934219000(800),100,168.93421(2)
69-Tm-170,169.935807100(800),,168.93421(2)
69-Tm-171,170.93643520(100),,168.93421(2)
69-Tm-172,171.93840700(600),,168.93421(2)
69-Tm-173,172.93960700(500),,168.93421(2)
69-Tm-174,173.9421700(500),,168.93421(2)
69-Tm-175,174.9438400(500),,168.93421(2)
69-Tm-176,175.947000(110),,168.93421(2)
69-Tm-177,176.948930(220)#,,168.93421(2)
69-Tm-178,177.952510(320)#,,168.93421(2)
69-Tm-179,178.955020(430)#,,168.93421(2)
70-Yb-148,147.967550(430)#,,173.04(3)
70-Yb-149,148.964220(320)#,,173.04(3)
70-Yb-150,149.958310(320)#,,173.04(3)
70-Yb-151,150.955400(320),,173.04(3)
70-Yb-152,151.950330(160),,173.04(3)
70-Yb-153,152.949370(220)#,,173.04(3)
70-Yb-154,153.9463960(190),,173.04(3)
70-Yb-155,154.9457830(180),,173.04(3)
70-Yb-156,155.9428170(100),,173.04(3)
70-Yb-157,156.9426510(120),,173.04(3)
70-Yb-158,157.93987100(900),,173.04(3)
70-Yb-159,158.9400600(190),,173.04(3)
70-Yb-160,159.93755900(600),,173.04(3)
70-Yb-161,160.9379120(160),,173.04(3)
70-Yb-162,161.9357790(160),,173.04(3)
70-Yb-163,162.9363450(160),,173.04(3)
70-Yb-164,163.9345010(160),,173.04(3)
70-Yb-165,164.9352700(280),,173.04(3)
70-Yb-166,165.93387600(800),,173.04(3)
70-Yb-167,166.93495400(400),,173.04(3)
70-Yb-168,167.933891300(100),0.13(1),173.04(3)
70-Yb-169,168.935184210(190),,173.04(3)
70-Yb-170,169.9347672430(110),3.04(15),173.04(3)
70-Yb-171,170.9363315150(140),14.28(57),173.04(3)
70-Yb-172,171.9363866540(150),21.83(67),173.04(3)
70-Yb-173,172.9382162120(120),16.13(27),173.04(3)
70-Yb-174,173.9388675460(120),31.83(92),173.04(3)
70-Yb-175,174.9412819100(800),,173.04(3)
70-Yb-176,175.9425747060(160),12.76(41),173.04(3)
70-Yb-177,176.945263850(240),,173.04(3)
70-Yb-178,177.94666900(700),,173.04(3)
70-Yb-179,178.949930(220)#,,173.04(3)
70-Yb-180,179.951990(320)#,,173.04(3)
70-Yb-181,180.955890(320)#,,173.04(3)
71-Lu-150,149.973410(320)#,,174.967(1)
71-Lu-151,150.967470(320)#,,174.967(1)
71-Lu-152,151.964120(210)#,,174.967(1)
71-Lu-153,152.958800(160),,174.967(1)
71-Lu-154,153.957420(220)#,,174.967(1)
71-Lu-155,154.9543260(210),,174.967(1)
71-Lu-156,155.9530900(600),,174.967(1)
71-Lu-157,156.9501450(130),,174.967(1)
71-Lu-158,157.9493160(160),,174.967(1)
71-Lu-159,158.9466400(400),,174.967(1)
71-Lu-160,159.9460300(600),,174.967(1)
71-Lu-161,160.9435700(300),,174.967(1)
71-Lu-162,161.9432800(800),,174.967(1)
71-Lu-163,162.9411800(300),,174.967(1)
71-Lu-164,163.9413400(300),,174.967(1)
71-Lu-165,164.9394070(280),,174.967(1)
71-Lu-166,165.9398600(300),,174.967(1)
71-Lu-167,166.9382400(400),,174.967(1)
71-Lu-168,167.9387300(400),,174.967(1)
71-Lu-169,168.93764600(300),,174.967(1)
71-Lu-170,169.9384790(180),,174.967(1)
71-Lu-171,170.93791860(200),,174.967(1)
71-Lu-172,171.93909130(250),,174.967(1)
71-Lu-173,172.93893570(170),,174.967(1)
71-Lu-174,173.94034280(170),,174.967(1)
71-Lu-175,174.94077720(130),97.41(2),174.967(1)
71-Lu-176,175.94269170(130),2.59(2),174.967(1)
71-Lu-177,176.94376360(130),,174.967(1)
71-Lu-178,177.94596010(240),,174.967(1)
71-Lu-179,178.94733300(600),,174.967(1)
71-Lu-180,179.9498900(800),,174.967(1)
71-Lu-181,180.951910(140),,174.967(1)
71-Lu-182,181.955160(220)#,,174.967(1)
71-Lu-183,182.9573600(900),,174.967(1)
71-Lu-184,183.961030(220)#,,174.967(1)
72-Hf-154,153.964860(320)#,,178.49(2)
72-Hf-155,154.963170(320)#,,178.49(2)
72-Hf-156,155.959400(160),,178.49(2)
72-Hf-157,156.958290(220)#,,178.49(2)
72-Hf-158,157.9548010(190),,178.49(2)
72-Hf-159,158.9539960(180),,178.49(2)
72-Hf-160,159.9506830(100),,178.49(2)
72-Hf-161,160.9502780(250),,178.49(2)
72-Hf-162,161.9472160(100),,178.49(2)
72-Hf-163,162.9471070(280),,178.49(2)
72-Hf-164,163.9443710(170),,178.49(2)
72-Hf-165,164.9445700(300),,178.49(2)
72-Hf-166,165.9421800(300),,178.49(2)
72-Hf-167,166.9426000(300),,178.49(2)
72-Hf-168,167.9405700(300),,178.49(2)
72-Hf-169,168.9412600(300),,178.49(2)
72-Hf-170,169.9396100(300),,178.49(2)
72-Hf-171,170.9404900(300),,178.49(2)
72-Hf-172,171.9394500(260),,178.49(2)
72-Hf-173,172.9405100(300),,178.49(2)
72-Hf-174,173.94004840(240),0.16(1),178.49(2)
72-Hf-175,174.94151140(250),,178.49(2)
72-Hf-176,175.94140980(160),5.26(7),178.49(2)
72-Hf-177,176.94323020(150),18.60(9),178.49(2)
72-Hf-178,177.94370830(150),27.28(7),178.49(2)
72-Hf-179,178.94582570(150),13.62(2),178.49(2)
72-Hf-180,179.94655950(150),35.08(16),178.49(2)
72-Hf-181,180.94911080(150),,178.49(2)
72-Hf-182,181.95056400(700),,178.49(2)
72-Hf-183,182.9535300(300),,178.49(2)
72-Hf-184,183.9554500(400),,178.49(2)
72-Hf-185,184.9588600(700),,178.49(2)
72-Hf-186,185.9609000(600),,178.49(2)
73-Ta-156,155.972090(320)#,,180.9479(1)
73-Ta-157,156.968230(160),,180.9479(1)
73-Ta-158,157.966590(220)#,,180.9479(1)
73-Ta-159,158.9630280(210),,180.9479(1)
73-Ta-160,159.9615400(600),,180.9479(1)
73-Ta-161,160.9583690(260),,180.9479(1)
73-Ta-162,161.9572900(700),,180.9479(1)
73-Ta-163,162.9543400(400),,180.9479(1)
73-Ta-164,163.9535300(300),,180.9479(1)
73-Ta-165,164.9507800(150),,180.9479(1)
73-Ta-166,165.9505100(300),,180.9479(1)
73-Ta-167,166.9480900(300),,180.9479(1)
73-Ta-168,167.9480500(300),,180.9479(1)
73-Ta-169,168.9460100(300),,180.9479(1)
73-Ta-170,169.9461800(300),,180.9479(1)
73-Ta-171,170.9444800(300),,180.9479(1)
73-Ta-172,171.9449000(300),,180.9479(1)
73-Ta-173,172.9437500(300),,180.9479(1)
73-Ta-174,173.9444500(300),,180.9479(1)
73-Ta-175,174.9437400(300),,180.9479(1)
73-Ta-176,175.9448600(300),,180.9479(1)
73-Ta-177,176.94448200(400),,180.9479(1)
73-Ta-178,177.9456800(600)#,,180.9479(1)
73-Ta-179,178.94593910(160),,180.9479(1)
73-Ta-180,179.94746760(220),0.012(2),180.9479(1)
73-Ta-181,180.94799850(170),99.988(2),180.9479(1)
73-Ta-182,181.95015460(170),,180.9479(1)
73-Ta-183,182.95137540(170),,180.9479(1)
73-Ta-184,183.9540100(280),,180.9479(1)
73-Ta-185,184.9555610(150),,180.9479(1)
73-Ta-186,185.9585500(600),,180.9479(1)
73-Ta-187,186.9603900(600),,180.9479(1)
73-Ta-188,187.963600(220)#,,180.9479(1)
74-W-158,157.974570(320)#,,183.84(1)
74-W-159,158.972700(320)#,,183.84(1)
74-W-160,159.968510(160),,183.84(1)
74-W-161,160.967250(220)#,,183.84(1)
74-W-162,161.9635000(190),,183.84(1)
74-W-163,162.9625200(600),,183.84(1)
74-W-164,163.9589520(100),,183.84(1)
74-W-165,164.9582810(280),,183.84(1)
74-W-166,165.9550320(100),,183.84(1)
74-W-167,166.9548110(200),,183.84(1)
74-W-168,167.9518050(140),,183.84(1)
74-W-169,168.9517790(170),,183.84(1)
74-W-170,169.9492310(140),,183.84(1)
74-W-171,170.9494500(300),,183.84(1)
74-W-172,171.9472900(300),,183.84(1)
74-W-173,172.9476900(300),,183.84(1)
74-W-174,173.9460800(300),,183.84(1)
74-W-175,174.9467200(300),,183.84(1)
74-W-176,175.9456300(300),,183.84(1)
74-W-177,176.9466400(300),,183.84(1)
74-W-178,177.9458860(160),,183.84(1)
74-W-179,178.9470790(160),,183.84(1)
74-W-180,179.94671330(150),0.12(1),183.84(1)
74-W-181,180.94821870(160),,183.84(1)
74-W-182,181.948205600(800),26.50(16),183.84(1)
74-W-183,182.950224400(800),14.31(4),183.84(1)
74-W-184,183.950933200(800),30.64(2),183.84(1)
74-W-185,184.953421200(800),,183.84(1)
74-W-186,185.95436510(130),28.43(19),183.84(1)
74-W-187,186.95716120(130),,183.84(1)
74-W-188,187.95848800(300),,183.84(1)
74-W-189,188.961560(220)#,,183.84(1)
74-W-190,189.9631000(400),,183.84(1)
75-Re-160,159.981880(320)#,,186.207(1)
75-Re-161,160.977620(160),,186.207(1)
75-Re-162,161.975900(220)#,,186.207(1)
75-Re-163,162.9720850(200),,186.207(1)
75-Re-164,163.9705100(600),,186.207(1)
75-Re-165,164.9670860(250),,186.207(1)
75-Re-166,165.9658200(900),,186.207(1)
75-Re-167,166.9626000(400)#,,186.207(1)
75-Re-168,167.9615700(300),,186.207(1)
75-Re-169,168.9587660(120),,186.207(1)
75-Re-170,169.9582350(120),,186.207(1)
75-Re-171,170.9557200(300),,186.207(1)
75-Re-172,171.9553800(400),,186.207(1)
75-Re-173,172.9532400(300),,186.207(1)
75-Re-174,173.9531200(300),,186.207(1)
75-Re-175,174.9513800(300),,186.207(1)
75-Re-176,175.9516200(300),,186.207(1)
75-Re-177,176.9503300(300),,186.207(1)
75-Re-178,177.9509900(300),,186.207(1)
75-Re-179,178.9499900(260),,186.207(1)
75-Re-180,179.9507920(230),,186.207(1)
75-Re-181,180.9500620(130),,186.207(1)
75-Re-182,181.951210(110),,186.207(1)
75-Re-183,182.95082100(900),,186.207(1)
75-Re-184,183.95252800(500),,186.207(1)
75-Re-185,184.952958300(900),37.40(2),186.207(1)
75-Re-186,185.954989200(900),,186.207(1)
75-Re-187,186.955752200(800),62.60(2),186.207(1)
75-Re-188,187.958113700(800),,186.207(1)
75-Re-189,188.95922800(900),,186.207(1)
75-Re-190,189.96180000(500),,186.207(1)
75-Re-191,190.9631230(110),,186.207(1)
75-Re-192,191.9660900(800),,186.207(1)
76-Os-162,161.984430(320)#,,190.23(3)
76-Os-163,162.982460(320)#,,190.23(3)
76-Os-164,163.978070(160),,190.23(3)
76-Os-165,164.976650(220)#,,190.23(3)
76-Os-166,165.9726980(190),,190.23(3)
76-Os-167,166.9715500(900),,190.23(3)
76-Os-168,167.9677990(110),,190.23(3)
76-Os-169,168.9670180(280),,190.23(3)
76-Os-170,169.9635790(100),,190.23(3)
76-Os-171,170.9631800(200),,190.23(3)
76-Os-172,171.9600170(140),,190.23(3)
76-Os-173,172.9598080(160),,190.23(3)
76-Os-174,173.9570630(110),,190.23(3)
76-Os-175,174.9569450(130),,190.23(3)
76-Os-176,175.9547700(120),,190.23(3)
76-Os-177,176.9549580(160),,190.23(3)
76-Os-178,177.9532530(150),,190.23(3)
76-Os-179,178.9538160(170),,190.23(3)
76-Os-180,179.9523820(170),,190.23(3)
76-Os-181,180.9532470(270),,190.23(3)
76-Os-182,181.9521100(230),,190.23(3)
76-Os-183,182.9531300(500),,190.23(3)
76-Os-184,183.952492900(900),0.02(1),190.23(3)
76-Os-185,184.954046000(900),,190.23(3)
76-Os-186,185.953837600(800),1.59(3),190.23(3)
76-Os-187,186.955749600(800),1.96(2),190.23(3)
76-Os-188,187.955837300(800),13.24(8),190.23(3)
76-Os-189,188.958145900(700),16.15(5),190.23(3)
76-Os-190,189.958445400(700),26.26(2),190.23(3)
76-Os-191,190.960928100(700),,190.23(3)
76-Os-192,191.96147880(250),40.78(19),190.23(3)
76-Os-193,192.96414960(250),,190.23(3)
76-Os-194,193.96517940(260),,190.23(3)
76-Os-195,194.9683200(600),,190.23(3)
76-Os-196,195.9696400(400),,190.23(3)
77-Ir-165,164.987550(170)#,,192.217(3)
77-Ir-166,165.985720(220)#,,192.217(3)
77-Ir-167,166.9816720(200),,192.217(3)
77-Ir-168,167.9799600(600),,192.217(3)
77-Ir-169,168.9762820(250),,192.217(3)
77-Ir-170,169.975110(110)#,,192.217(3)
77-Ir-171,170.9716500(400),,192.217(3)
77-Ir-172,171.9706100(300),,192.217(3)
77-Ir-173,172.9675050(110),,192.217(3)
77-Ir-174,173.9669500(120),,192.217(3)
77-Ir-175,174.9641500(130),,192.217(3)
77-Ir-176,175.96362600(900),,192.217(3)
77-Ir-177,176.9613020(210),,192.217(3)
77-Ir-178,177.9610790(200),,192.217(3)
77-Ir-179,178.9591180(100),,192.217(3)
77-Ir-180,179.9592290(230),,192.217(3)
77-Ir-181,180.95763500(600),,192.217(3)
77-Ir-182,181.9580760(230),,192.217(3)
77-Ir-183,182.9568410(260),,192.217(3)
77-Ir-184,183.9574800(300),,192.217(3)
77-Ir-185,184.9567000(300),,192.217(3)
77-Ir-186,185.9579470(180),,192.217(3)
77-Ir-187,186.9575400(300),,192.217(3)
77-Ir-188,187.9588350(100),,192.217(3)
77-Ir-189,188.9587230(140),,192.217(3)
77-Ir-190,189.96054340(150),,192.217(3)
77-Ir-191,190.96059150(140),37.3(2),192.217(3)
77-Ir-192,191.96260240(140),,192.217(3)
77-Ir-193,192.96292380(140),62.7(2),192.217(3)
77-Ir-194,193.96507570(140),,192.217(3)
77-Ir-195,194.96597690(140),,192.217(3)
77-Ir-196,195.9684000(400),,192.217(3)
77-Ir-197,196.9696570(220),,192.217(3)
77-Ir-198,197.972400(220)#,,192.217(3)
77-Ir-199,198.9738100(400),,192.217(3)
78-Pt-168,167.988180(160),,195.078(2)
78-Pt-169,168.986620(220)#,,195.078(2)
78-Pt-170,169.9825020(200),,195.078(2)
78-Pt-171,170.9812500(900),,195.078(2)
78-Pt-172,171.9773410(110),,195.078(2)
78-Pt-173,172.9764500(700),,195.078(2)
78-Pt-174,173.9728200(110),,195.078(2)
78-Pt-175,174.9724010(200),,195.078(2)
78-Pt-176,175.9689380(140),,195.078(2)
78-Pt-177,176.9684700(160),,195.078(2)
78-Pt-178,177.9656490(110),,195.078(2)
78-Pt-179,178.96535900(900),,195.078(2)
78-Pt-180,179.9630380(110),,195.078(2)
78-Pt-181,180.9630900(150),,195.078(2)
78-Pt-182,181.9611720(140),,195.078(2)
78-Pt-183,182.9615960(150),,195.078(2)
78-Pt-184,183.9599220(160),,195.078(2)
78-Pt-185,184.9606140(280),,195.078(2)
78-Pt-186,185.9593510(230),,195.078(2)
78-Pt-187,186.9606170(260),,195.078(2)
78-Pt-188,187.95939800(600),,195.078(2)
78-Pt-189,188.9608480(110),,195.078(2)
78-Pt-190,189.959949800(700),0.014(1),195.078(2)
78-Pt-191,190.96167600(400),,195.078(2)
78-Pt-192,191.96104270(280),0.782(7),195.078(2)
78-Pt-193,192.96298450(150),,195.078(2)
78-Pt-194,193.962683500(500),32.967(99),195.078(2)
78-Pt-195,194.964794300(500),33.832(10),195.078(2)
78-Pt-196,195.964954600(500),25.242(41),195.078(2)
78-Pt-197,196.967343000(600),,195.078(2)
78-Pt-198,197.96789670(230),7.163(55),195.078(2)
78-Pt-199,198.97059700(230),,195.078(2)
78-Pt-200,199.9714450(220),,195.078(2)
78-Pt-201,200.9745100(500),,195.078(2)
78-Pt-202,201.9756390(270),,195.078(2)
79-Au-171,170.9918820(220),,196.96655(2)
79-Au-172,171.9900000(600),,196.96655(2)
79-Au-173,172.9862240(240),,196.96655(2)
79-Au-174,173.984910(110)#,,196.96655(2)
79-Au-175,174.9813200(400),,196.96655(2)
79-Au-176,175.9801200(400),,196.96655(2)
79-Au-177,176.9768700(110),,196.96655(2)
79-Au-178,177.9760570(110),,196.96655(2)
79-Au-179,178.9731740(130),,196.96655(2)
79-Au-180,179.97249000(500),,196.96655(2)
79-Au-181,180.9700790(210),,196.96655(2)
79-Au-182,181.9696140(200),,196.96655(2)
79-Au-183,182.9675880(100),,196.96655(2)
79-Au-184,183.9674520(240),,196.96655(2)
79-Au-185,184.96579890(280),,196.96655(2)
79-Au-186,185.9659530(230),,196.96655(2)
79-Au-187,186.9645420(240),,196.96655(2)
79-Au-188,187.96524800(290),,196.96655(2)
79-Au-189,188.9639480(220),,196.96655(2)
79-Au-190,189.96475200(400),,196.96655(2)
79-Au-191,190.96371600(500),,196.96655(2)
79-Au-192,191.9648180(170),,196.96655(2)
79-Au-193,192.96413800(900),,196.96655(2)
79-Au-194,193.96541910(230),,196.96655(2)
79-Au-195,194.96503780(120),,196.96655(2)
79-Au-196,195.96657100(300),,196.96655(2)
79-Au-197,196.966570100(600),100,196.96655(2)
79-Au-198,197.968243700(600),,196.96655(2)
79-Au-199,198.968766600(600),,196.96655(2)
79-Au-200,199.9707570(290),,196.96655(2)
79-Au-201,200.97165800(300),,196.96655(2)
79-Au-202,201.9738560(250),,196.96655(2)
79-Au-203,202.97515400(300),,196.96655(2)
79-Au-204,203.978110(220)#,,196.96655(2)
79-Au-205,204.980060(220)#,,196.96655(2)
80-Hg-175,174.9914400(900),,200.59(2)
80-Hg-176,175.9873490(120),,200.59(2)
80-Hg-177,176.9862800(900),,200.59(2)
80-Hg-178,177.9824850(120),,200.59(2)
80-Hg-179,178.9818200(300),,200.59(2)
80-Hg-180,179.9782600(140),,200.59(2)
80-Hg-181,180.9778190(170),,200.59(2)
80-Hg-182,181.9746890(110),,200.59(2)
80-Hg-183,182.97444500(800),,200.59(2)
80-Hg-184,183.9717180(100),,200.59(2)
80-Hg-185,184.9718910(150),,200.59(2)
80-Hg-186,185.9693620(130),,200.59(2)
80-Hg-187,186.9698140(140),,200.59(2)
80-Hg-188,187.96758100(700),,200.59(2)
80-Hg-189,188.9681900(300),,200.59(2)
80-Hg-190,189.9663220(170),,200.59(2)
80-Hg-191,190.9671580(240),,200.59(2)
80-Hg-192,191.9656340(170),,200.59(2)
80-Hg-193,192.9666530(170),,200.59(2)
80-Hg-194,193.96544900(300),,200.59(2)
80-Hg-195,194.9667060(250),,200.59(2)
80-Hg-196,195.96583300(300),0.15(1),200.59(2)
80-Hg-197,196.96721400(300),,200.59(2)
80-Hg-198,197.966769200(500),9.97(20),200.59(2)
80-Hg-199,198.968281000(600),16.87(22),200.59(2)
80-Hg-200,199.968326900(600),23.10(19),200.59(2)
80-Hg-201,200.970303100(800),13.18(9),200.59(2)
80-Hg-202,201.970643600(800),29.86(26),200.59(2)
80-Hg-203,202.97287240(180),,200.59(2)
80-Hg-204,203.973494000(500),6.87(15),200.59(2)
80-Hg-205,204.97607300(400),,200.59(2)
80-Hg-206,205.9775140(220),,200.59(2)
80-Hg-207,206.9823000(300),,200.59(2)
80-Hg-208,207.9857600(300),,200.59(2)
81-Tl-177,176.9964140(230),,204.3833(2)
81-Tl-178,177.995050(110)#,,204.3833(2)
81-Tl-179,178.9911200(400),,204.3833(2)
81-Tl-180,179.9899200(800),,204.3833(2)
81-Tl-181,180.9862600(100),,204.3833(2)
81-Tl-182,181.9856930(130),,204.3833(2)
81-Tl-183,182.9821930(100),,204.3833(2)
81-Tl-184,183.9818750(110),,204.3833(2)
81-Tl-185,184.9787890(220),,204.3833(2)
81-Tl-186,185.9786550(220),,204.3833(2)
81-Tl-187,186.97590500(900),,204.3833(2)
81-Tl-188,187.9760200(300),,204.3833(2)
81-Tl-189,188.97357400(900),,204.3833(2)
81-Tl-190,189.97384200(800),,204.3833(2)
81-Tl-191,190.97178400(800),,204.3833(2)
81-Tl-192,191.9722300(300),,204.3833(2)
81-Tl-193,192.97050200(700),,204.3833(2)
81-Tl-194,193.9710810(150),,204.3833(2)
81-Tl-195,194.9697740(120),,204.3833(2)
81-Tl-196,195.9704810(130),,204.3833(2)
81-Tl-197,196.9695600(150),,204.3833(2)
81-Tl-198,197.97044700(800),,204.3833(2)
81-Tl-199,198.9698800(300),,204.3833(2)
81-Tl-200,199.97096400(600),,204.3833(2)
81-Tl-201,200.9708200(150),,204.3833(2)
81-Tl-202,201.97210890(200),,204.3833(2)
81-Tl-203,202.97234410(130),29.524(14),204.3833(2)
81-Tl-204,203.97386340(120),,204.3833(2)
81-Tl-205,204.97442730(130),70.476(14),204.3833(2)
81-Tl-206,205.97611010(140),,204.3833(2)
81-Tl-207,206.97741900(600),,204.3833(2)
81-Tl-208,207.98201800(200),,204.3833(2)
81-Tl-209,208.98535200(700),,204.3833(2)
81-Tl-210,209.9900730(120),,204.3833(2)
82-Pb-181,180.9966600(900),,207.2(1)
82-Pb-182,181.9926740(130),,207.2(1)
82-Pb-183,182.9918600(300),,207.2(1)
82-Pb-184,183.9881360(140),,207.2(1)
82-Pb-185,184.9876100(170),,207.2(1)
82-Pb-186,185.9842390(120),,207.2(1)
82-Pb-187,186.98391100(500),,207.2(1)
82-Pb-188,187.9808790(110),,207.2(1)
82-Pb-189,188.9808440(150),,207.2(1)
82-Pb-190,189.9780820(130),,207.2(1)
82-Pb-191,190.97821600(700),,207.2(1)
82-Pb-192,191.97579000(600),,207.2(1)
82-Pb-193,192.9761360(110),,207.2(1)
82-Pb-194,193.9740120(190),,207.2(1)
82-Pb-195,194.97451600(500),,207.2(1)
82-Pb-196,195.97278800(800),,207.2(1)
82-Pb-197,196.97343500(500),,207.2(1)
82-Pb-198,197.97201500(900),,207.2(1)
82-Pb-199,198.97291300(700),,207.2(1)
82-Pb-200,199.9718190(110),,207.2(1)
82-Pb-201,200.9728700(150),,207.2(1)
82-Pb-202,201.97215200(400),,207.2(1)
82-Pb-203,202.97339100(700),,207.2(1)
82-Pb-204,203.97304350(120),1.4(1),207.2(1)
82-Pb-205,204.97448170(120),,207.2(1)
82-Pb-206,205.97446520(120),24.1(1),207.2(1)
82-Pb-207,206.97589680(120),22.1(1),207.2(1)
82-Pb-208,207.97665200(120),52.4(1),207.2(1)
82-Pb-209,208.98109000(190),,207.2(1)
82-Pb-210,209.98418840(160),,207.2(1)
82-Pb-211,210.98873530(240),,207.2(1)
82-Pb-212,211.99189590(200),,207.2(1)
82-Pb-213,212.99656100(700),,207.2(1)
82-Pb-214,213.99980350(210),,207.2(1)
83-Bi-185,184.9976000(900)#,,208.98038(2)
83-Bi-186,185.9966230(180),,208.98038(2)
83-Bi-187,186.9931470(110),,208.98038(2)
83-Bi-188,187.9922760(120),,208.98038(2)
83-Bi-189,188.9891950(220),,208.98038(2)
83-Bi-190,189.9886250(230),,208.98038(2)
83-Bi-191,190.98578700(800),,208.98038(2)
83-Bi-192,191.9854700(300),,208.98038(2)
83-Bi-193,192.98294700(800),,208.98038(2)
83-Bi-194,193.98279900(600),,208.98038(2)
83-Bi-195,194.98064900(600),,208.98038(2)
83-Bi-196,195.9806670(260),,208.98038(2)
83-Bi-197,196.97886500(900),,208.98038(2)
83-Bi-198,197.9792010(300),,208.98038(2)
83-Bi-199,198.9776730(110),,208.98038(2)
83-Bi-200,199.9781310(240),,208.98038(2)
83-Bi-201,200.9769950(130),,208.98038(2)
83-Bi-202,201.9777230(150),,208.98038(2)
83-Bi-203,202.9768920(140),,208.98038(2)
83-Bi-204,203.9778360(100),,208.98038(2)
83-Bi-205,204.97738500(500),,208.98038(2)
83-Bi-206,205.97849900(800),,208.98038(2)
83-Bi-207,206.97847060(260),,208.98038(2)
83-Bi-208,207.97974210(250),,208.98038(2)
83-Bi-209,208.98039860(150),100,208.98038(2)
83-Bi-210,209.98412020(150),,208.98038(2)
83-Bi-211,210.98726900(600),,208.98038(2)
83-Bi-212,211.99128500(200),,208.98038(2)
83-Bi-213,212.99438400(500),,208.98038(2)
83-Bi-214,213.9987110(120),,208.98038(2)
83-Bi-215,215.00174900(600),,208.98038(2)
83-Bi-216,216.0063060(120),,208.98038(2)
84-Po-190,189.9951020(140),,[209]
84-Po-191,190.99455800(800),,[209]
84-Po-192,191.9913400(110),,[209]
84-Po-193,192.9910620(160),,[209]
84-Po-194,193.9881860(140),,[209]
84-Po-195,194.98806600(600),,[209]
84-Po-196,195.98554100(600),,[209]
84-Po-197,196.9856220(110),,[209]
84-Po-198,197.9833890(190),,[209]
84-Po-199,198.98364000(600),,[209]
84-Po-200,199.98181200(800),,[209]
84-Po-201,200.98226400(500),,[209]
84-Po-202,201.98073900(900),,[209]
84-Po-203,202.98141600(500),,[209]
84-Po-204,203.9803100(110),,[209]
84-Po-205,204.9811900(110),,[209]
84-Po-206,205.98047400(400),,[209]
84-Po-207,206.98159300(700),,[209]
84-Po-208,207.98124600(180),,[209]
84-Po-209,208.98243040(190),,[209]
84-Po-210,209.98287370(120),,[209]
84-Po-211,210.98665320(130),,[209]
84-Po-212,211.98886800(120),,[209]
84-Po-213,212.99285700(300),,[209]
84-Po-214,213.99520130(160),,[209]
84-Po-215,214.99941840(230),,[209]
84-Po-216,216.00191340(190),,[209]
84-Po-217,217.00631600(700),,[209]
84-Po-218,218.00897120(210),,[209]
85-At-193,192.9999280(230),,[210]
85-At-194,193.9992310(250),,[210]
85-At-195,194.9962740(100),,[210]
85-At-196,195.9958000(300),,[210]
85-At-197,196.99317700(900),,[210]
85-At-198,197.99279800(500),,[210]
85-At-199,198.99052800(600),,[210]
85-At-200,199.9903510(260),,[210]
85-At-201,200.98841700(900),,[210]
85-At-202,201.9886260(300),,[210]
85-At-203,202.9869430(110),,[210]
85-At-204,203.9872510(240),,[210]
85-At-205,204.9860610(130),,[210]
85-At-206,205.9866460(150),,[210]
85-At-207,206.9858000(130),,[210]
85-At-208,207.9866130(100),,[210]
85-At-209,208.98616900(500),,[210]
85-At-210,209.98714700(800),,[210]
85-At-211,210.98749620(290),,[210]
85-At-212,211.99073730(260),,[210]
85-At-213,212.99293700(500),,[210]
85-At-214,213.99637200(400),,[210]
85-At-215,214.99865100(700),,[210]
85-At-216,216.00242300(400),,[210]
85-At-217,217.00471800(500),,[210]
85-At-218,218.0086960(120),,[210]
85-At-219,219.01116100(300),,[210]
85-At-220,220.0154330(150),,[210]
85-At-221,221.0180170(150),,[210]
85-At-222,222.0224940(170),,[210]
85-At-223,223.0251510(150),,[210]
86-Rn-196,196.0021200(150),,[222]
86-Rn-197,197.0016210(170),,[222]
86-Rn-198,197.9986790(140),,[222]
86-Rn-199,198.99832500(800),,[222]
86-Rn-200,199.99570500(600),,[222]
86-Rn-201,200.9955910(110),,[222]
86-Rn-202,201.9932640(190),,[222]
86-Rn-203,202.99336100(600),,[222]
86-Rn-204,203.99144400(800),,[222]
86-Rn-205,204.99172300(500),,[222]
86-Rn-206,205.99019500(900),,[222]
86-Rn-207,206.99073000(500),,[222]
86-Rn-208,207.9896350(110),,[222]
86-Rn-209,208.9904010(110),,[222]
86-Rn-210,209.98968900(500),,[222]
86-Rn-211,210.99060100(700),,[222]
86-Rn-212,211.99070400(300),,[222]
86-Rn-213,212.99388500(400),,[222]
86-Rn-214,213.9953630(100),,[222]
86-Rn-215,214.99874500(700),,[222]
86-Rn-216,216.00027200(600),,[222]
86-Rn-217,217.00392800(500),,[222]
86-Rn-218,218.00560110(250),,[222]
86-Rn-219,219.00947870(230),,[222]
86-Rn-220,220.01139240(190),,[222]
86-Rn-221,221.01553600(600),,[222]
86-Rn-222,222.01757600(210),,[222]
86-Rn-223,223.02188900(800),,[222]
86-Rn-224,224.0240960(110),,[222]
86-Rn-225,225.0284860(120),,[222]
86-Rn-226,226.0308610(110),,[222]
86-Rn-227,227.0353040(150),,[222]
86-Rn-228,228.0378350(190),,[222]
87-Fr-200,200.0065800(300),,[223]
87-Fr-201,201.0038520(100),,[223]
87-Fr-202,202.00333000(600),,[223]
87-Fr-203,203.00094100(700),,[223]
87-Fr-204,204.0006520(260),,[223]
87-Fr-205,204.99859400(800),,[223]
87-Fr-206,205.9986610(300),,[223]
87-Fr-207,206.9969410(190),,[223]
87-Fr-208,207.9971390(130),,[223]
87-Fr-209,208.9959400(120),,[223]
87-Fr-210,209.9964110(140),,[223]
87-Fr-211,210.9955550(130),,[223]
87-Fr-212,211.99622500(900),,[223]
87-Fr-213,212.99618400(500),,[223]
87-Fr-214,213.99897100(900),,[223]
87-Fr-215,215.00034200(800),,[223]
87-Fr-216,216.00319000(400),,[223]
87-Fr-217,217.00463200(700),,[223]
87-Fr-218,218.00757900(500),,[223]
87-Fr-219,219.00925100(700),,[223]
87-Fr-220,220.01232700(400),,[223]
87-Fr-221,221.01425400(500),,[223]
87-Fr-222,222.01758300(800),,[223]
87-Fr-223,223.01973420(210),,[223]
87-Fr-224,224.0233480(120),,[223]
87-Fr-225,225.0255720(130),,[223]
87-Fr-226,226.02954500(700),,[223]
87-Fr-227,227.03186500(600),,[223]
87-Fr-228,228.03583900(700),,[223]
87-Fr-229,229.03829100(500),,[223]
87-Fr-230,230.04239100(700),,[223]
87-Fr-231,231.04517500(800),,[223]
87-Fr-232,232.0494610(150),,[223]
88-Ra-203,203.0092340(100),,[226]
88-Ra-204,204.0065070(100),,[226]
88-Ra-205,205.0062310(240),,[226]
88-Ra-206,206.0038280(190),,[226]
88-Ra-207,207.0037700(600),,[226]
88-Ra-208,208.0018550(100),,[226]
88-Ra-209,209.00199500(600),,[226]
88-Ra-210,210.0004750(100),,[226]
88-Ra-211,211.00089300(500),,[226]
88-Ra-212,211.9997870(110),,[226]
88-Ra-213,213.0003710(110),,[226]
88-Ra-214,214.00010000(600),,[226]
88-Ra-215,215.00271800(800),,[226]
88-Ra-216,216.00353400(900),,[226]
88-Ra-217,217.00632300(800),,[226]
88-Ra-218,218.0071340(110),,[226]
88-Ra-219,219.01008500(700),,[226]
88-Ra-220,220.01102800(800),,[226]
88-Ra-221,221.01391700(500),,[226]
88-Ra-222,222.01537300(500),,[226]
88-Ra-223,223.01850060(220),,[226]
88-Ra-224,224.02021040(190),,[226]
88-Ra-225,225.02361050(280),,[226]
88-Ra-226,226.02540820(210),,[226]
88-Ra-227,227.02917620(210),,[226]
88-Ra-228,228.03106860(210),,[226]
88-Ra-229,229.0349570(170),,[226]
88-Ra-230,230.0370550(110),,[226]
88-Ra-231,231.0410270(120),,[226]
88-Ra-232,232.0434750(100),,[226]
88-Ra-233,233.04759500(900),,[226]
88-Ra-234,234.05038200(900),,[226]
89-Ac-207,207.0119700(600),,[227]
89-Ac-208,208.0115500(700),,[227]
89-Ac-209,209.0095000(600),,[227]
89-Ac-210,210.0094100(700),,[227]
89-Ac-211,211.0076700(600),,[227]
89-Ac-212,212.0078360(230),,[227]
89-Ac-213,213.0065930(130),,[227]
89-Ac-214,214.0069060(150),,[227]
89-Ac-215,215.0064740(130),,[227]
89-Ac-216,216.0087490(100),,[227]
89-Ac-217,217.0093420(120),,[227]
89-Ac-218,218.0116500(600),,[227]
89-Ac-219,219.0124200(600),,[227]
89-Ac-220,220.01475500(700),,[227]
89-Ac-221,221.0156000(600),,[227]
89-Ac-222,222.01784400(500),,[227]
89-Ac-223,223.01913600(700),,[227]
89-Ac-224,224.02172200(400),,[227]
89-Ac-225,225.02322900(500),,[227]
89-Ac-226,226.02609700(300),,[227]
89-Ac-227,227.02775060(210),,[227]
89-Ac-228,228.03101970(220),,[227]
89-Ac-229,229.0329470(130),,[227]
89-Ac-230,230.0363270(170),,[227]
89-Ac-231,231.0383930(140),,[227]
89-Ac-232,232.0420340(140),,[227]
89-Ac-233,233.0443460(140),,[227]
89-Ac-234,234.0481390(150),,[227]
89-Ac-235,235.0508400(150),,[227]
89-Ac-236,236.0549900(400),,[227]
90-Th-210,210.0150940(200),,232.0381(1)
90-Th-211,211.0149000(900),,232.0381(1)
90-Th-212,212.0130020(110),,232.0381(1)
90-Th-213,213.0130110(100),,232.0381(1)
90-Th-214,214.0114810(110),,232.0381(1)
90-Th-215,215.01172500(700),,232.0381(1)
90-Th-216,216.0110560(120),,232.0381(1)
90-Th-217,217.0131030(110),,232.0381(1)
90-Th-218,218.0132760(110),,232.0381(1)
90-Th-219,219.0155300(600),,232.0381(1)
90-Th-220,220.0157700(150),,232.0381(1)
90-Th-221,221.01818600(900),,232.0381(1)
90-Th-222,222.0184680(110),,232.0381(1)
90-Th-223,223.02081100(900),,232.0381(1)
90-Th-224,224.0214660(100),,232.0381(1)
90-Th-225,225.02395100(500),,232.0381(1)
90-Th-226,226.02490400(500),,232.0381(1)
90-Th-227,227.02770250(220),,232.0381(1)
90-Th-228,228.02873970(190),,232.0381(1)
90-Th-229,229.03176140(260),,232.0381(1)
90-Th-230,230.03313230(130),,232.0381(1)
90-Th-231,231.03630280(130),,232.0381(1)
90-Th-232,232.03805360(150),100,232.0381(1)
90-Th-233,233.04158010(150),,232.0381(1)
90-Th-234,234.04359980(280),,232.0381(1)
90-Th-235,235.0472550(140),,232.0381(1)
90-Th-236,236.0496570(150),,232.0381(1)
90-Th-237,237.0536290(170),,232.0381(1)
90-Th-238,238.056390(300)#,,232.0381(1)
91-Pa-213,213.0211000(600),,231.03588(2)
91-Pa-214,214.0208900(900),,231.03588(2)
91-Pa-215,215.0191100(900),,231.03588(2)
91-Pa-216,216.0191350(260),,231.03588(2)
91-Pa-217,217.0183090(130),,231.03588(2)
91-Pa-218,218.0200210(190),,231.03588(2)
91-Pa-219,219.0199500(700),,231.03588(2)
91-Pa-220,220.0217700(160),,231.03588(2)
91-Pa-221,221.0218700(600),,231.03588(2)
91-Pa-222,222.0236900(900),,231.03588(2)
91-Pa-223,223.0239800(800),,231.03588(2)
91-Pa-224,224.02561700(800),,231.03588(2)
91-Pa-225,225.0261500(900),,231.03588(2)
91-Pa-226,226.0279480(120),,231.03588(2)
91-Pa-227,227.02880400(800),,231.03588(2)
91-Pa-228,228.03105100(500),,231.03588(2)
91-Pa-229,229.03209600(400),,231.03588(2)
91-Pa-230,230.03454000(300),,231.03588(2)
91-Pa-231,231.03588250(190),100,231.03588(2)
91-Pa-232,232.03859000(800),,231.03588(2)
91-Pa-233,233.04024650(140),,231.03588(2)
91-Pa-234,234.04330600(400),,231.03588(2)
91-Pa-235,235.0453990(150),,231.03588(2)
91-Pa-236,236.0486680(150),,231.03588(2)
91-Pa-237,237.0510230(140),,231.03588(2)
91-Pa-238,238.0546370(170),,231.03588(2)
91-Pa-239,239.057260(210)#,,231.03588(2)
91-Pa-240,240.061200(220)#,,231.03588(2)
92-U-218,218.0235050(150),,238.02891(3)
92-U-219,219.0250090(140),,238.02891(3)
92-U-220,220.024710(110)#,,238.02891(3)
92-U-221,221.0263200(800),,238.02891(3)
92-U-222,222.0260600(600),,238.02891(3)
92-U-223,223.0279600(600),,238.02891(3)
92-U-224,224.0276360(160),,238.02891(3)
92-U-225,225.0293850(110),,238.02891(3)
92-U-226,226.0293390(120),,238.02891(3)
92-U-227,227.03118100(900),,238.02891(3)
92-U-228,228.0313690(140),,238.02891(3)
92-U-229,229.03350600(600),,238.02891(3)
92-U-230,230.03394000(500),,238.02891(3)
92-U-231,231.03629220(290),,238.02891(3)
92-U-232,232.03715480(190),,238.02891(3)
92-U-233,233.03963430(240),,238.02891(3)
92-U-234,234.04095030(120),0.0055(2),238.02891(3)
92-U-235,235.04392810(120),0.7200(51),238.02891(3)
92-U-236,236.04556610(120),,238.02891(3)
92-U-237,237.04872830(130),,238.02891(3)
92-U-238,238.05078690(160),99.2745(106),238.02891(3)
92-U-239,239.05429200(160),,238.02891(3)
92-U-240,240.05659240(270),,238.02891(3)
92-U-241,241.060330(210)#,,238.02891(3)
92-U-242,242.062930(220)#,,238.02891(3)
93-Np-225,225.033940(100),,[237]
93-Np-226,226.035230(110),,[237]
93-Np-227,227.0349800(800),,[237]
93-Np-228,228.036310(110)#,,[237]
93-Np-229,229.036290(110),,[237]
93-Np-230,230.0378300(600),,[237]
93-Np-231,231.0382400(500),,[237]
93-Np-232,232.040110(110)#,,[237]
93-Np-233,233.0407400(500),,[237]
93-Np-234,234.04289300(900),,[237]
93-Np-235,235.04406150(150),,[237]
93-Np-236,236.0465700(500),,[237]
93-Np-237,237.04817160(120),,[237]
93-Np-238,238.05094460(120),,[237]
93-Np-239,239.05293750(140),,[237]
93-Np-240,240.0561640(180),,[237]
93-Np-241,241.058310(110),,[237]
93-Np-242,242.061640(210),,[237]
93-Np-243,243.0642000(300)#,,[237]
93-Np-244,244.067890(110)#,,[237]
94-Pu-228,228.0387630(250),,[244]
94-Pu-229,229.0401500(700),,[244]
94-Pu-230,230.0396480(160),,[244]
94-Pu-231,231.0411260(240),,[244]
94-Pu-232,232.0411820(180),,[244]
94-Pu-233,233.0430000(600),,[244]
94-Pu-234,234.04331700(700),,[244]
94-Pu-235,235.0452850(220),,[244]
94-Pu-236,236.04605670(190),,[244]
94-Pu-237,237.04840790(180),,[244]
94-Pu-238,238.04955820(120),,[244]
94-Pu-239,239.05216160(120),,[244]
94-Pu-240,240.05381170(120),,[244]
94-Pu-241,241.05684970(120),,[244]
94-Pu-242,242.05874100(130),,[244]
94-Pu-243,243.06200210(270),,[244]
94-Pu-244,244.06420440(250),,[244]
94-Pu-245,245.0678250(150),,[244]
94-Pu-246,246.0702040(160),,[244]
94-Pu-247,247.074300(220)#,,[244]
95-Am-231,231.045530(320)#,,[243]
95-Am-232,232.046610(320)#,,[243]
95-Am-233,233.046470(120)#,,[243]
95-Am-234,234.047730(170)#,,[243]
95-Am-235,235.0479100(600),,[243]
95-Am-236,236.049430(130)#,,[243]
95-Am-237,237.0500000(600)#,,[243]
95-Am-238,238.0519800(600),,[243]
95-Am-239,239.05302270(210),,[243]
95-Am-240,240.0552980(150),,[243]
95-Am-241,241.05682730(120),,[243]
95-Am-242,242.05954740(120),,[243]
95-Am-243,243.06137990(150),,[243]
95-Am-244,244.06428290(160),,[243]
95-Am-245,245.06645280(200),,[243]
95-Am-246,246.0697740(190)#,,[243]
95-Am-247,247.072090(110)#,,[243]
95-Am-248,248.075750(220)#,,[243]
95-Am-249,249.078480(320)#,,[243]
96-Cm-233,233.0507700(900),,[247]
96-Cm-234,234.0501590(180),,[247]
96-Cm-235,235.051550(110)#,,[247]
96-Cm-236,236.0513720(190),,[247]
96-Cm-237,237.0528700(800),,[247]
96-Cm-238,238.0530820(130),,[247]
96-Cm-239,239.054910(160),,[247]
96-Cm-240,240.05552820(200),,[247]
96-Cm-241,241.05765120(170),,[247]
96-Cm-242,242.05883420(120),,[247]
96-Cm-243,243.06138730(160),,[247]
96-Cm-244,244.06275060(120),,[247]
96-Cm-245,245.06549100(120),,[247]
96-Cm-246,246.06722200(160),,[247]
96-Cm-247,247.07035300(400),,[247]
96-Cm-248,248.07234910(250),,[247]
96-Cm-249,249.07595400(250),,[247]
96-Cm-250,250.0783580(110),,[247]
96-Cm-251,251.0822850(240),,[247]
96-Cm-252,252.084870(320)#,,[247]
97-Bk-235,235.056650(430)#,,[247]
97-Bk-236,236.057480(390)#,,[247]
97-Bk-237,237.057120(250)#,,[247]
97-Bk-238,238.058200(280)#,,[247]
97-Bk-239,239.058240(220)#,,[247]
97-Bk-240,240.059760(160)#,,[247]
97-Bk-241,241.060100(180)#,,[247]
97-Bk-242,242.062000(140)#,,[247]
97-Bk-243,243.06300600(500),,[247]
97-Bk-244,244.0651790(150),,[247]
97-Bk-245,245.06635980(190),,[247]
97-Bk-246,246.0686700(600),,[247]
97-Bk-247,247.07030600(600),,[247]
97-Bk-248,248.0731400(500),,[247]
97-Bk-249,249.07498310(130),,[247]
97-Bk-250,250.07831700(300),,[247]
97-Bk-251,251.0807610(120),,[247]
97-Bk-252,252.084310(220)#,,[247]
97-Bk-253,253.086880(390)#,,[247]
97-Bk-254,254.090600(320)#,,[247]
98-Cf-237,237.062200(100),,[251]
98-Cf-238,238.061490(320)#,,[251]
98-Cf-239,239.062480(130)#,,[251]
98-Cf-240,240.0622530(190),,[251]
98-Cf-241,241.063690(180)#,,[251]
98-Cf-242,242.0637550(140),,[251]
98-Cf-243,243.065480(190)#,,[251]
98-Cf-244,244.06599940(280),,[251]
98-Cf-245,245.06804680(260),,[251]
98-Cf-246,246.06880370(160),,[251]
98-Cf-247,247.0709710(150),,[251]
98-Cf-248,248.07218300(500),,[251]
98-Cf-249,249.07485040(130),,[251]
98-Cf-250,250.07640450(170),,[251]
98-Cf-251,251.07958700(400),,[251]
98-Cf-252,252.08162650(250),,[251]
98-Cf-253,253.08513400(500),,[251]
98-Cf-254,254.0873240(120),,[251]
98-Cf-255,255.091050(220)#,,[251]
98-Cf-256,256.093440(340)#,,[251]
99-Es-240,240.068950(390)#,,[252]
99-Es-241,241.068590(250)#,,[252]
99-Es-242,242.069570(280)#,,[252]
99-Es-243,243.069510(220)#,,[252]
99-Es-244,244.070880(200)#,,[252]
99-Es-245,245.071190(180)#,,[252]
99-Es-246,246.072810(100),,[252]
99-Es-247,247.0736220(210),,[252]
99-Es-248,248.0754700(600)#,,[252]
99-Es-249,249.0764100(300)#,,[252]
99-Es-250,250.078610(110)#,,[252]
99-Es-251,251.07999100(600),,[252]
99-Es-252,252.0829800(500),,[252]
99-Es-253,253.08482120(130),,[252]
99-Es-254,254.08802400(300),,[252]
99-Es-255,255.0902740(120),,[252]
99-Es-256,256.093600(110)#,,[252]
99-Es-257,257.095980(440)#,,[252]
100-Fm-242,242.073430(430)#,,[257]
100-Fm-243,243.074410(140)#,,[257]
100-Fm-244,244.074040(220)#,,[257]
100-Fm-245,245.075350(210)#,,[257]
100-Fm-246,246.0753530(150),,[257]
100-Fm-247,247.076940(190)#,,[257]
100-Fm-248,248.07718500(900),,[257]
100-Fm-249,249.07892600(700),,[257]
100-Fm-250,250.07952000(800),,[257]
100-Fm-251,251.0815450(150),,[257]
100-Fm-252,252.08246600(600),,[257]
100-Fm-253,253.08518090(170),,[257]
100-Fm-254,254.08685240(200),,[257]
100-Fm-255,255.08996300(400),,[257]
100-Fm-256,256.09177200(300),,[257]
100-Fm-257,257.09510500(500),,[257]
100-Fm-258,258.097080(220)#,,[257]
100-Fm-259,259.100600(300)#,,[257]
101-Md-245,245.080860(280)#,,[258]
101-Md-246,246.081710(280)#,,[258]
101-Md-247,247.081520(220)#,,[258]
101-Md-248,248.082610(200)#,,[258]
101-Md-249,249.082860(180),,[258]
101-Md-250,250.084160(100),,[258]
101-Md-251,251.0847740(200),,[258]
101-Md-252,252.086390(100),,[258]
101-Md-253,253.0871400(300)#,,[258]
101-Md-254,254.089590(110)#,,[258]
101-Md-255,255.09108200(600),,[258]
101-Md-256,256.093890(130)#,,[258]
101-Md-257,257.09553730(170),,[258]
101-Md-258,258.09843400(400),,[258]
101-Md-259,259.100450(110)#,,[258]
101-Md-260,260.103650(340)#,,[258]
102-No-249,249.087800(300)#,,[259]
102-No-250,250.087570(220)#,,[259]
102-No-251,251.088940(190)#,,[259]
102-No-252,252.0889660(100),,[259]
102-No-253,253.09056300(700),,[259]
102-No-254,254.0909540(100),,[259]
102-No-255,255.0931960(150),,[259]
102-No-256,256.09428200(800),,[259]
102-No-257,257.09688400(700),,[259]
102-No-258,258.098210(110)#,,[259]
102-No-259,259.10099800(700),,[259]
102-No-260,260.102640(220)#,,[259]
102-No-261,261.105700(220)#,,[259]
102-No-262,262.107460(390)#,,[259]
103-Lr-251,251.094290(220)#,,[262]
103-Lr-252,252.095050(200)#,,[262]
103-Lr-253,253.095030(180),,[262]
103-Lr-254,254.096240(100),,[262]
103-Lr-255,255.0965620(190),,[262]
103-Lr-256,256.0984900(900),,[262]
103-Lr-257,257.0994800(500)#,,[262]
103-Lr-258,258.101750(110)#,,[262]
103-Lr-259,259.1029000(800)#,,[262]
103-Lr-260,260.105500(130)#,,[262]
103-Lr-261,261.106880(220)#,,[262]
103-Lr-262,262.109620(220)#,,[262]
103-Lr-263,263.111290(240)#,,[262]
104-Rf-253,253.100530(440)#,,[261]
104-Rf-254,254.100060(300)#,,[261]
104-Rf-255,255.101270(190)#,,[261]
104-Rf-256,256.1011510(190),,[261]
104-Rf-257,257.1029170(120),,[261]
104-Rf-258,258.1034300(170),,[261]
104-Rf-259,259.1056000(800)#,,[261]
104-Rf-260,260.106440(220)#,,[261]
104-Rf-261,261.1087700(700),,[261]
104-Rf-262,262.109920(240)#,,[261]
104-Rf-263,263.112460(160)#,,[261]
104-Rf-264,264.113880(390)#,,[261]
105-Db-255,255.106920(300)#,,[262]
105-Db-256,256.107670(200)#,,[262]
105-Db-257,257.107520(180),,[262]
105-Db-258,258.108970(100),,[262]
105-Db-259,259.1094900(600),,[262]
105-Db-260,260.111300(100)#,,[262]
105-Db-261,261.111980(120)#,,[262]
105-Db-262,262.114070(150)#,,[262]
105-Db-263,263.114990(180)#,,[262]
105-Db-264,264.117300(250)#,,[262]
105-Db-265,265.118500(240)#,,[262]
106-Sg-258,258.113040(440)#,,[266]
106-Sg-259,259.114350(190)#,,[266]
106-Sg-260,260.1143830(220),,[266]
106-Sg-261,261.1159480(200),,[266]
106-Sg-262,262.1163390(240),,[266]
106-Sg-263,263.118300(100)#,,[266]
106-Sg-264,264.118930(300)#,,[266]
106-Sg-265,265.121090(150)#,,[266]
106-Sg-266,266.121970(260)#,,[266]
107-Bh-260,260.121440(210)#,,[264]
107-Bh-261,261.121400(190),,[264]
107-Bh-262,262.122650(100),,[264]
107-Bh-263,263.122920(330)#,,[264]
107-Bh-264,264.124490(190)#,,[264]
107-Bh-265,265.124960(260)#,,[264]
107-Bh-266,266.126790(180)#,,[264]
107-Bh-267,267.127500(280)#,,[264]
108-Hs-263,263.128480(210)#,,[277]
108-Hs-264,264.1283600(300),,[277]
108-Hs-265,265.1297920(260),,[277]
108-Hs-266,266.1300490(290),,[277]
108-Hs-267,267.131680(100)#,,[277]
108-Hs-268,268.132010(320)#,,[277]
108-Hs-269,269.133650(140)#,,[277]
108-Hs-277,277.151770(480)#,,[277]
109-Mt-265,265.135940(470)#,,[268]
109-Mt-266,266.137060(100),,[268]
109-Mt-267,267.137190(540)#,,[268]
109-Mt-268,268.138650(250)#,,[268]
109-Mt-269,269.138810(340)#,,[268]
109-Mt-270,270.140320(210)#,,[268]
109-Mt-271,271.140740(350)#,,[268]
110-Ds-267,267.143730(220)#,,[281]
110-Ds-268,268.143480(320)#,,[281]
110-Ds-269,269.1447500(300),,[281]
110-Ds-270,270.1445900(400),,[281]
110-Ds-271,271.145950(100)#,,[281]
110-Ds-272,272.146090(460)#,,[281]
110-Ds-273,273.148460(150)#,,[281]
110-Ds-281,281.164550(530)#,,[281]
111-Rg-272,272.153270(250)#,,[272]
112-Cn-285,285.177230(540)#,,[285]
113-Nh-286,286.182460(630)#,,[286]
114-Fl-289,289.190520(550)#,,[289]
115-Mc-289,289.193970(830)#,,[289]
116-Lv-293,293.204580(550)#,,[293]
117-Ts-294,294.210840(640)#,,[294]
118-Og-294,294.213980(590)#,,[294]\
"""
