// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_WEBUI_ASH_ADD_SUPERVISION_ADD_SUPERVISION_METRICS_RECORDER_H_
#define CHROME_BROWSER_UI_WEBUI_ASH_ADD_SUPERVISION_ADD_SUPERVISION_METRICS_RECORDER_H_

#include "base/memory/raw_ref.h"
#include "base/time/time.h"

namespace base {
class TickClock;
}

// Records UMA metrics for users going through the Add Supervision process.
class AddSupervisionMetricsRecorder {
 public:
  // These enum values represent the state that the user has attained while
  // going through the Add Supervision dialog.
  // These values are persisted to logs. Entries should not be renumbered and
  // numeric values should never be reused.
  enum class EnrollmentState {
    // Recorded when user opens Add Supervision dialog.
    kInitiated = 0,
    // Recorded when user successfully enrolls in supervision.
    kCompleted = 1,
    // Recorded when user clicks "Sign out" after enrollment in the dialog.
    kSignedOut = 2,
    // Recorded when user closes the dialog without enrollment, excluding sign
    // out.
    kClosed = 3,
    // Recorded when user signs out to switch accounts.
    kSwitchedAccounts = 4,
    // Add future entries above this comment, in sync with enums.xml.
    // Update kMaxValue to the last value.
    kMaxValue = kSwitchedAccounts
  };

  AddSupervisionMetricsRecorder(const AddSupervisionMetricsRecorder&) = delete;
  AddSupervisionMetricsRecorder& operator=(
      const AddSupervisionMetricsRecorder&) = delete;

  static AddSupervisionMetricsRecorder* GetInstance();

  // Records UMA metrics for users going through the Add Supervision process.
  void RecordAddSupervisionEnrollment(EnrollmentState action);

  // For testing purposes only: RAII helper to set the clock to a dummy clock
  // for testing, and restore the original clock afterwards.
  class ScopedClockForTesting {
   public:
    ScopedClockForTesting(AddSupervisionMetricsRecorder& recorder,
                          const base::TickClock& clock);
    ~ScopedClockForTesting();

   private:
    raw_ref<AddSupervisionMetricsRecorder> recorder_;
    raw_ref<const base::TickClock> original_clock_;
  };

 private:
  AddSupervisionMetricsRecorder();

  // Records UMA metric of how long the user spends in the Add Supervision
  // process in milliseconds.
  void RecordUserTime(const char* metric_name) const;

  // Points to the base::DefaultTickClock by default.
  raw_ref<const base::TickClock> clock_;

  // Records when the user initiates the Add Supervision process.
  base::TimeTicks start_time_;
};
#endif  // CHROME_BROWSER_UI_WEBUI_ASH_ADD_SUPERVISION_ADD_SUPERVISION_METRICS_RECORDER_H_
